package com.ekingwin.bas.cloud.service.impl;


import com.ekingwin.bas.cloud.dao.IpGoodsCategoryMapper;
import com.ekingwin.bas.cloud.dto.GoodsCategoryTreeDto;
import com.ekingwin.bas.cloud.dto.IpGoodsCategoryDto;
import com.ekingwin.bas.cloud.dto.SaveGoodsCategoryDto;
import com.ekingwin.bas.cloud.handler.BusinessException;
import com.ekingwin.bas.cloud.service.IClassifyServie;
import com.ekingwin.bas.cloud.util.IdUtil;
import com.ekingwin.bas.cloud.util.StringUtils;
import com.ekingwin.bas.cloud.vo.*;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author  zhangjie
 */
@Service
@Primary
@Transactional(rollbackFor = Exception.class)
public class ClassifyServiceImpl implements IClassifyServie {


    @Autowired
    private IpGoodsCategoryMapper ipCategoryMapper;

    @Value("${dataSourceName}")
    private String basUrl;
    /**
     * 新增商品时获取物资三级类别
     * @param ipGoodsCategoryDto
     * @return
     */
    @Override
    public PageInfo getThirdClassify(IpGoodsCategoryDto ipGoodsCategoryDto) {
        PageHelper.startPage(ipGoodsCategoryDto.getPagenum(), ipGoodsCategoryDto.getPagesize());
        List<IpGoodsCategoryVo> ipGoodsCategoryVoList = ipCategoryMapper.getThirdClassify(ipGoodsCategoryDto);
        PageInfo pageInfo = new PageInfo<>(ipGoodsCategoryVoList);
        return pageInfo;
    }

    /**
     * 首页导航栏获取商品三级分类
     * @param ipGoodsCategoryDto
     * @return
     * @throws Exception
     */
    @Override
    public List<IpGoodsCategoryDetailVo> getThirdClassifyBy(IpGoodsCategoryDto ipGoodsCategoryDto) throws Exception {
        String categoryType = ipGoodsCategoryDto.getCategoryType();
        //判断是否存在categoryType类型
        if (StringUtils.isNotEmpty(categoryType)) {
            ipGoodsCategoryDto.setCustomerType(categoryType);
        }
        List<IpGoodsCategoryDetailVo> classify = ipCategoryMapper.getThirdClassifyBy(ipGoodsCategoryDto);
        return classify;
    }

    /**
     * 查看每一级的分类详情
     * @param categoryId
     * @return
     * @throws Exception
     */
    @Override
    public IpThirdCategoryInfoVo getGoodsCategoryInfo(String categoryId) throws Exception {
        return ipCategoryMapper.getGoodsCategoryInfo(categoryId);
    }

    /**
     * 新增修改分类信息
     * @param saveGoodsCategoryDto
     * @throws Exception
     */
    @Override
    public void saveOrUpdateCategoryInfo(SaveGoodsCategoryDto saveGoodsCategoryDto) throws Exception {
        String goodsCategoryid = saveGoodsCategoryDto.getId();
        //判断是否是新增、还是更新
        if (StringUtils.isEmpty(goodsCategoryid)) {
            //获取编码
            String code = saveGoodsCategoryDto.getCode();
            //判断编码是否存在，不存在则自动生成
            if (StringUtils.isEmpty(code)) {
                Long codeLong = IdUtil.nextId();
                code = codeLong.toString();
            }
            //判断分类编码是否重复
            int count = ipCategoryMapper.getCategoryCodeCount(code);
            if (count > 0) {
                throw new BusinessException(500, "分类编码重复");
            }
            saveGoodsCategoryDto.setCode(code);
            ipCategoryMapper.insertGoodsCategory(saveGoodsCategoryDto);
            String categoryId = saveGoodsCategoryDto.getId();
            //保存分类区域
            List<IpGoodSupplierAreaVo> areaVos = saveGoodsCategoryDto.getArea();
            if (areaVos != null && areaVos.size() > 0) {
                for (IpGoodSupplierAreaVo ipGoodSupplierAreaVo : areaVos) {
                    ipGoodSupplierAreaVo.setCategoryid(categoryId);
                }
                ipCategoryMapper.batchInsertCategorySup(areaVos);
            }
            //保存价格信息
            List<IpCategoryGoodsPriceVo> goodsPriceVos = saveGoodsCategoryDto.getSalesPrice();
            if (goodsPriceVos != null && goodsPriceVos.size() > 0) {
                for (IpCategoryGoodsPriceVo goodsPriceVo : goodsPriceVos) {
                    goodsPriceVo.setCategoryid(categoryId);
                }
                ipCategoryMapper.batchInsertCategoryPrice(goodsPriceVos);
            }
        } else {
            ipCategoryMapper.updateGoodsCategory(saveGoodsCategoryDto);
            ipCategoryMapper.deleteCategorySup(goodsCategoryid);
            ipCategoryMapper.deleteCategoryPrice(goodsCategoryid);
            //保存销售区域
            List<IpGoodSupplierAreaVo> areaVos = saveGoodsCategoryDto.getArea();
            if (areaVos != null && areaVos.size() > 0) {
                for (IpGoodSupplierAreaVo ipGoodSupplierAreaVo : areaVos) {
                    ipGoodSupplierAreaVo.setCategoryid(goodsCategoryid);
                }
                ipCategoryMapper.batchInsertCategorySup(areaVos);
            }
            //保存分类设置的价格
            List<IpCategoryGoodsPriceVo> goodsPriceVos = saveGoodsCategoryDto.getSalesPrice();
            if (goodsPriceVos != null && goodsPriceVos.size() > 0) {
                for (IpCategoryGoodsPriceVo goodsPriceVo : goodsPriceVos) {
                    goodsPriceVo.setCategoryid(goodsCategoryid);
                }
                ipCategoryMapper.batchInsertCategoryPrice(goodsPriceVos);
            }

        }

    }

    /**
     * 删除分类信息
     * @param categoryId
     * @throws Exception
     */
    @Override
    public void deleteCategoryInfo(String categoryId) throws Exception {
        //查询分类下是否有商品
        int count = ipCategoryMapper.getGoodsCountByCategoryCode(categoryId);
        if (count > 0) {
            throw new BusinessException(500, "该分类下存在商品,不允许删除");
        }
        int count1 = ipCategoryMapper.deleteGoodsCategory(categoryId);
        if (count1 > 0) {
            ipCategoryMapper.deleteCategorySup(categoryId);
        }
    }

    @Override
    public List<IpGoodsCategoryDetailVo> getGoodsCategoryInfoByLevel(IpGoodsCategoryDto ipGoodsCategoryDto) throws Exception {
        String type = ipGoodsCategoryDto.getType();
        List<IpGoodsCategoryDetailVo> classify = null;
        //0 一级分类、1二级分类 、2 三级分类
        if ("0".equals(type)) {
            classify = ipCategoryMapper.getGoodsCategoryInfoByLevel(ipGoodsCategoryDto);
        } else if ("1".equals(type)) {
            classify = ipCategoryMapper.getGoodsCategoryInfoByLevel1(ipGoodsCategoryDto);
        } else if ("2".equals(type)) {
            classify = ipCategoryMapper.getGoodsCategoryInfoByLevel2(ipGoodsCategoryDto);
        }

        return classify;
    }

    /**
     * 获取分类所在组织
     *
     * @param categoryTreeDto
     * @return
     */
    @Override
    public List<GoodsCategoryTreeVo> getGoodsCategoryTree(GoodsCategoryTreeDto categoryTreeDto) {
        String isSupplier = categoryTreeDto.getIsSupplier();
        String categoryId = categoryTreeDto.getCategoryId();

        List<IpGoodSupplierAreaVo> areaInfoOrSupplierId = null;
        //判断是否是供应商
        if (StringUtils.isNotEmpty(isSupplier) && "Y".equals(isSupplier)) {
            areaInfoOrSupplierId = ipCategoryMapper.getAreaInfoOrSupplierId(categoryId, categoryTreeDto.getUserid());
        } else {
            areaInfoOrSupplierId = ipCategoryMapper.getAreaInfoOrSupplierId(categoryId, null);
        }
        Set<String> treeHash = new HashSet<>();


        for (IpGoodSupplierAreaVo ipGoodSupplierAreaVo : areaInfoOrSupplierId) {
            String salesarea = ipGoodSupplierAreaVo.getSalesarea();
            String[] split = salesarea.split(",");
            for (String s : split) {
                String companyParent = ipCategoryMapper.getCompanyParent(s);
                String[] split1 = companyParent.split(",");
                boolean flag = true;
                for (String s1 : split1) {
                    if (treeHash.contains(s1)) {
                        flag = false;
                        break;
                    }
                }
                if (flag) {
                    treeHash.add(s);
                }
            }
        }
        //判断是否多个销售区域，若为多个去除上下级关系
        Set<String> endHash = new HashSet<>();
        //包含全集团的标识
        if (treeHash.contains("E")) {
            endHash.add("E");
        } else {
            endHash = this.getNoAllCompany(treeHash);
        }

        String areas = String.join(",", endHash);
        List<GoodsCategoryTreeVo> companyList = ipCategoryMapper.getCompanyList(areas);
        if (companyList != null) {
            for (GoodsCategoryTreeVo goodsCategoryTreeVo : companyList) {
                String orgtype = goodsCategoryTreeVo.getOrgtype();
                //若为集团、城市、区域则为文件
                if ("S".equals(orgtype) || "A".equals(orgtype) || "C".equals(orgtype)) {
                    goodsCategoryTreeVo.setParent(true);
                } else {
                    goodsCategoryTreeVo.setParent(false);
                }
            }
        }
        return companyList;
    }

    @Override
    public List<Map<String, Object>> getAllGoodsCategInfo(String companyid) {
        List<Map<String, Object>> goodsCategAuthority = ipCategoryMapper.getGoodsCategAuthority(basUrl,companyid);
        String salesareas = "";
        if(goodsCategAuthority!=null && goodsCategAuthority.size()>0){
            salesareas=goodsCategAuthority.get(0).get("ID")==null?"":goodsCategAuthority.get(0).get("ID")+"";
        }
        List<Map<String, Object>> allGoodsCategInfo = ipCategoryMapper.getAllGoodsCategInfo(basUrl,salesareas);
        if(allGoodsCategInfo!=null && allGoodsCategInfo.size()>0){
            return allGoodsCategInfo;
        }

        return new ArrayList<>();
    }


    private Set<String> getNoAllCompany(Set<String> list) {
        Set<String> endHash = list;
        if (list != null && list.size() > 0) {
            for (String s : list) {
                String companyChild = ipCategoryMapper.getCompanyChild(s);
                String[] split = companyChild.split(",");
                for (String s1 : split) {
                    if (endHash.contains(s1)) {
                        if (!s1.equals(s)) {
                            endHash.remove(s1);
                        }
                    }
                }
            }
        }
        return endHash;
    }
}
