package com.ekingwin.bas.cloud.service.impl;


import com.ekingwin.bas.cloud.dao.*;
import com.ekingwin.bas.cloud.dto.*;
import com.ekingwin.bas.cloud.entity.*;
import com.ekingwin.bas.cloud.handler.BusinessException;
import com.ekingwin.bas.cloud.infra.service.BaseService;
import com.ekingwin.bas.cloud.service.*;
import com.ekingwin.bas.cloud.util.*;
import com.ekingwin.bas.cloud.vo.*;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import net.sf.json.JSONObject;
import org.apache.logging.log4j.core.util.UuidUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.*;

/**
 * The type Ip goods service.
 *
 * @program: rcworkspace
 * @description:
 * @author: zhangjie
 * @create: 2019 -06-27 16:12
 */
@Service
@Primary
@Transactional(rollbackFor = Exception.class)
public class IpGoodsServiceImpl extends BaseService implements IpGoodsService {

    private Logger logger = LoggerFactory.getLogger(IpGoodsServiceImpl.class);


    @Autowired
    private IpGoodsMapper goodsMapper;

    @Autowired
    private IpGoodsSupplierFreightMapper supplierFreightMapper;

    @Autowired
    private IpGoodsPriceMapper ipGoodsPriceMapper;

    @Autowired
    private IpGoodsImgMapper ipGoodsImgMapper;

    @Autowired
    private IpGoodsLogsMapper ipGoodsLogsMapper;

    @Autowired
    private IpGoodsSuppliersMapper ipGoodsSuppliersMapper;

    @Autowired
    private IpCategoryMapper ipCategoryMapper;

    @Value("${goods.detailurl}")
    private String filePath;

    @Value("${goods.baseurl}")
    private String baseurl;

    @Value("${goods.baseurls}")
    private String baseurls;

    /**
     * 商品下架
     */
    @Value("${goods.solr.deleteData}")
    private String solrDeleteData;

    @Value("${goods.solr.addData}")
    private String solrAddData;

    @Value("${goods.solr.initsalenum}")
    private String initsalenum;


    @Autowired
    private InitSolrDataService initSolrDataService;

    @Autowired
    private IpGoodsDraftService ipGoodsDraftService;

    @Autowired
    private IpGoodsSapService ipGoodsSapService;


    @Autowired
    private IpGoodProcessMapper ipGoodProcessMapper;

    @Autowired
    private IpGoodFootService ipGoodFootService;


    @Value("${dataSourceName}")
    private String basUrl;

    private InitSolrGoodsService initSolrGoodsService = null;

    /**
     * 保存商品信息
     * 1.先保存商品主表
     *
     * @return
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public Long saveGoods(GoodsAddDto goodsAddDto) throws Exception {

        //判断是否直接上架
        String directiveground = goodsAddDto.getDirectiveground();

        Integer status = goodsAddDto.getStatus();
        String goodsNum = "";
        if (StringUtils.isNotEmpty(goodsAddDto.getGoodsNum())) {
            goodsNum = goodsAddDto.getGoodsNum();
        } else {
            goodsNum = this.getGoodSsNum();
        }
        Long busskey = 0L;
        Long goodsId = goodsAddDto.getGoodsId();
        //若为直接上架
        if (StringUtils.isEmpty(goodsId) || StringUtils.isEmpty(goodsAddDto.getInstid())) {
            //保存主表信息
            busskey = this.saveOrUpdate(goodsAddDto, goodsNum);

            //保存商品价格信息
            this.saveOrUpdatePrice(busskey, goodsAddDto);

            //保存运费信息
            this.saveOrUpdateSupplierFreight(busskey, goodsAddDto);

            //保存商品图片
            this.saveOrUpdateFile(busskey, goodsAddDto);
            goodsAddDto.setGoodsId(busskey);
        }

        //判断是否是提交，若为提交则判断是否直接上架
        //如果直接上架则调用solr  添加商品,否则发起审批进行审批
        if (directiveground.equals("Y") && StringUtils.isEmpty(goodsAddDto.getInstid())) {
            //判断是否是提交，若提交则同步solr库
            if (status.equals(1)) {
                Map<String, Object> goodsMap = goodsAddDto.getGoodsMap();
                String categoryId = goodsMap.get("FTNPThreeCategoryCode").toString();
                this.updateGoodsCategory(categoryId);
                initSolrGoodsService = new InitSolrGoodsService();
                initSolrGoodsService.setSolrAddData(solrAddData);
                initSolrGoodsService.start(busskey, initSolrGoodsService);
            }

        } else {
            busskey = ipGoodsDraftService.saveGoodsProcess(goodsAddDto, goodsNum, busskey);
        }

        return busskey;
    }

    /**
     * 判断提交逻辑
     *
     * @param goodsAddDto
     */
    private void checkRule(GoodsAddDto goodsAddDto) {
        //校验1是否在有效期
        String directivegroundv = goodsAddDto.getDirectiveground();
        Map<String, Object> map = goodsAddDto.getGoodsMap();
        String successDate = map.get("successdate").toString();
        String name = map.get("name").toString();
        if (directivegroundv.equals("Y")) {

            String nowDateStr = DateUtils.format(new Date(), DateUtils.DATE_PATTERN);
            boolean flag = DateUtils.compareDate(successDate, nowDateStr);
            if (!flag) {
                throw new BusinessException(500, name + ",该商品有效期小于当前日期，不能上架！请编辑后再提交上架申请。");
            }
        }
    }


    /**
     * 保存商品主表信息
     *
     * @param goodsAddDto
     * @return
     */
    private Long saveOrUpdate(GoodsAddDto goodsAddDto, String goodsNum) throws Exception {
        String type = "编辑";
        /**
         * 判断是提交还是更新
         */
        Integer statusSun = goodsAddDto.getStatus();

        Long goodsId = goodsAddDto.getGoodsId();
        if (StringUtils.isEmpty(goodsId)) {
            type = "新增";
        }
        String detailFile = "";
        if (StringUtils.isNotEmpty(goodsAddDto.getDetails())) {
            //保存附件信息
            detailFile = "/" + UuidUtil.getTimeBasedUuid() + ".txt";
            //将2商品描述写入文件
            ReadTxtUtil.writeFile(filePath + detailFile, goodsAddDto.getDetails());
        }
        String descriptionFile = "";
        //保存规格参数
        if (StringUtils.isNotEmpty(goodsAddDto.getDescription())) {
            //保存附件信息
            descriptionFile = "/" + UuidUtil.getTimeBasedUuid() + ".txt";

            //将2商品描述写入文件
            ReadTxtUtil.writeFile(filePath + descriptionFile, goodsAddDto.getDescription());

        }
        //先保存主表信息
        InsertMap insertGoodsMap = new InsertMap();
        Map<String, Object> goodsMap = goodsAddDto.getGoodsMap();
        String ftnpjc = null;
        if (StringUtils.isNotEmpty(goodsMap.get("FTNPJC"))) {
            ftnpjc = goodsMap.get("FTNPJC").toString();
        }
        goodsMap.put("goodsnum", goodsNum);
        //判断是否直接上架
        String directiveground = goodsAddDto.getDirectiveground();

        Integer status = 4;
        if (StringUtils.isNotEmpty(directiveground)) {

            //直接上架处理
            if (directiveground.equals("Y") && statusSun.equals(1)) {
                type = "上架";
                status = 1;
            } else {
                status = 6;
            }
        }


        String allPrice = "0";
        if (StringUtils.isNotEmpty(goodsMap.get("allPrice"))) {
            allPrice = goodsMap.get("allPrice").toString();
        }
        goodsMap.put("details", detailFile);
        goodsMap.put("description", descriptionFile);
        goodsMap.put("state", status);
        goodsMap.put("price", allPrice);
        insertGoodsMap.setTableName(IpGoodsEntity.TABLENAME);
        goodsMap.put("directiveground", directiveground);
        goodsMap.put("urlid", goodsAddDto.getUrlid());
        goodsMap.put("userid", goodsAddDto.getUserid());

        insertGoodsMap.setParams(goodsMap);
        if (StringUtils.isEmpty(goodsId)) {
            goodsMap.put("inputdate", DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
            goodsMapper.insertGoods(insertGoodsMap);
            goodsId = insertGoodsMap.getId();
        } else {
            insertGoodsMap.setId(goodsId);
            goodsMapper.updateGoods(insertGoodsMap);
        }

        if (StringUtils.isNotEmpty(directiveground) && directiveground.equals("Y") && statusSun.equals(1)) {
            //同步sap
            if (StringUtils.isEmpty(ftnpjc) || "XM".equals(ftnpjc)) {
                this.initGoodsSap("03", goodsId.toString(), goodsAddDto);
            }
        }
        //记录操作日志
        this.saveGoodLog(goodsId, goodsAddDto, type);

        return goodsId;
    }

    /**
     * 保存商品费用
     */
    private void saveOrUpdatePrice(Long busskey, GoodsAddDto goodsAddDto) throws Exception {
        ipGoodsPriceMapper.updateByGoodsid(busskey);
        List<IpGoodsPriceDto> ipGoodsPriceDtos = goodsAddDto.getPriceDtoList();
        if (ipGoodsPriceDtos != null && ipGoodsPriceDtos.size() > 0) {
            IpGoodsPrice goodsPrice = null;
            for (IpGoodsPriceDto ipGoodsPriceDto : ipGoodsPriceDtos) {
                Long id = ipGoodsPriceDto.getId();
                goodsPrice = new IpGoodsPrice();
                MapObjUtil.copy(ipGoodsPriceDto, goodsPrice);
                goodsPrice.setGoodsId(busskey);
                goodsPrice.setIsDelete("N");
                int detailByGoodsidCount = 0;
                if (id != null && id != 0) {
                    detailByGoodsidCount = ipGoodsPriceMapper.getDetailByGoodsidCount(id, busskey);
                }
                if (detailByGoodsidCount > 0) {
                    ipGoodsPriceMapper.updateByPrimaryKeySelective(goodsPrice);
                } else {
                    goodsPrice.setId(null);
                    ipGoodsPriceMapper.insertSelective(goodsPrice);
                }

            }
        }
    }

    /**
     * 保存供应商运费
     */
    private void saveOrUpdateSupplierFreight(Long busskey, GoodsAddDto goodsAddDto) throws Exception {
        supplierFreightMapper.deleteByGoodsId(busskey);

        ipGoodsSuppliersMapper.deleteByGoodsId(busskey);
        //获取供应商信息
        List<IpGoodsSuppliersDto> suppliersDtoListList = goodsAddDto.getSupplierList();
        if (suppliersDtoListList != null && suppliersDtoListList.size() > 0) {
            IpGoodsSuppliers ipGoodsSuppliers = null;
            for (IpGoodsSuppliersDto ipGoodsSuppliersDto : suppliersDtoListList) {
                ipGoodsSuppliers = new IpGoodsSuppliers();
                MapObjUtil.copy(ipGoodsSuppliersDto, ipGoodsSuppliers);
                ipGoodsSuppliers.setGoodsId(busskey);
                ipGoodsSuppliersMapper.insertSelective(ipGoodsSuppliers);
                Long suppliersId = ipGoodsSuppliers.getId();
                //获取供应商运费信息
                List<IpGoodsSupplierFreightDto> ipGoodsPriceDtos = ipGoodsSuppliersDto.getSupplierFreightList();
                if (ipGoodsPriceDtos != null && ipGoodsPriceDtos.size() > 0) {
                    IpGoodsSupplierFreight ipGoodsSupplierFreight = null;

                    for (IpGoodsSupplierFreightDto ipGoodsSupplierFreightDto : ipGoodsPriceDtos) {
                        ipGoodsSupplierFreight = new IpGoodsSupplierFreight();
                        MapObjUtil.copy(ipGoodsSupplierFreightDto, ipGoodsSupplierFreight);
                        ipGoodsSupplierFreight.setGoodsId(busskey);
                        ipGoodsSupplierFreight.setGoodsSuppliersId(suppliersId);
                        supplierFreightMapper.insertSelective(ipGoodsSupplierFreight);
                    }
                }
            }
        }
    }

    /**
     * 保存商品图片
     *
     * @param busskey
     * @param goodsAddDto
     */
    private void saveOrUpdateFile(Long busskey, GoodsAddDto goodsAddDto) {
        ipGoodsImgMapper.deleteByGoodsId(busskey);
        if (StringUtils.isNotEmpty(goodsAddDto.getUrlid())) {
            IpGoodsImg ipGoodsImg = new IpGoodsImg();
            ipGoodsImg.setGoodsid(busskey);
            ipGoodsImg.setGoodsimgid(goodsAddDto.getUrlid());
            ipGoodsImgMapper.insertSelective(ipGoodsImg);
        }
    }


    private void saveGoodLog(Long busskey, GoodsAddDto goodsAddDto, String type) {

        IpGoodsLogsWithBLOBs logsWithBLOBs = new IpGoodsLogsWithBLOBs();
        logsWithBLOBs.setGoodsid(Integer.parseInt(busskey.toString()));
        logsWithBLOBs.setOperatetype(type);
        logsWithBLOBs.setOperatedate(DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
        Map<String, Object> map = goodsAddDto.getGoodsMap();
        String successDate = map.get("successdate").toString();
        logsWithBLOBs.setSuccessdate(successDate);
        if (StringUtils.isNotEmpty(map.get("allPrice"))) {
            String allPrice = map.get("allPrice").toString();
            logsWithBLOBs.setPrice(allPrice);
        } else {
            logsWithBLOBs.setPrice("0.00");
        }

        logsWithBLOBs.setSubmitpeople(goodsAddDto.getUserid());
        logsWithBLOBs.setSubmitpeoplename(goodsAddDto.getUserName());
        ipGoodsLogsMapper.insertSelective(logsWithBLOBs);
    }


    /**
     * 获取商品详情
     *
     * @param goodsGetDetailDto
     * @return
     */
    @Override
    public GoodsDeatilVo getGoodsDetail(GoodsGetDetailDto goodsGetDetailDto) throws Exception {
        GoodsDeatilVo goodsDeatilVo = null;
        if (StringUtils.isNotEmpty(goodsGetDetailDto.getGoodsId())) {
            goodsDeatilVo = this.getGoodsDetailByGoodsId(goodsGetDetailDto.getGoodsId(), null);
        } else {
            String instid = goodsGetDetailDto.getInstid();
            goodsDeatilVo = ipGoodsDraftService.getGoodsProcessByinstid(instid);
            if (goodsDeatilVo == null) {
                IpGoodProcess ipGoodProcess = ipGoodsDraftService.getGoodProcess(instid);
                goodsDeatilVo = this.getGoodsDetailByGoodsId(ipGoodProcess.getGoodsid(), ipGoodProcess.getGoodsnum());
            }

        }
        if (goodsDeatilVo != null) {
            Map<String, Object> goodsMap = goodsDeatilVo.getGoodsMap();
            if (StringUtils.isNotEmpty(goodsMap.get("supplierid"))) {
                String supplierid = goodsMap.get("supplierid").toString();
                GoodsSupplierInfoVo goodsSuppliersById = ipGoodsSuppliersMapper.getGoodsSuppliersById(supplierid);
                goodsSuppliersById.setSuppliernum(supplierid);
                goodsDeatilVo.setSupplierInfo(goodsSuppliersById);
            }

        }


        return goodsDeatilVo;
    }

    /**
     * Init foot service.
     *
     * @param goodsDeatilVo     the goods deatil vo
     * @param goodsGetDetailDto the goods get detail dto
     */
    @Async
    public void initFootService(GoodsDeatilVo goodsDeatilVo, GoodsGetDetailDto goodsGetDetailDto) {
        ipGoodFootService.saveGoodsFoot(goodsDeatilVo, goodsGetDetailDto);
    }

    /**
     * 获取商品详情
     *
     * @param goodId
     * @param goodsnum
     * @return
     */
    private Map<String, Object> getGoodsMap(Long goodId, String goodsnum) {
        Map<String, Object> map = null;
        if (StringUtils.isNotEmpty(goodsnum)) {
            map = goodsMapper.getGoodsDetailByNum(goodsnum);
        } else {
            map = goodsMapper.getGoodsDetail(goodId);
        }
        return map;
    }

    /**
     * 根据商品获取详情
     *
     * @param goodId
     * @return
     * @throws Exception
     */
    private GoodsDeatilVo getGoodsDetailByGoodsId(Long goodId, String goodsnum) throws Exception {
        GoodsDeatilVo goodsDeatilVo = new GoodsDeatilVo();
        Map<String, Object> map = this.getGoodsMap(goodId, goodsnum);


        //获取商品详情
        if (StringUtils.isNotEmpty(map.get("details"))) {
            String detailPath = map.get("details").toString();
            //获取商品1详情
            String detail = ReadTxtUtil.readFile(filePath + detailPath);
            if (!detail.contains(baseurl)) {
                detail = detail.replaceAll(baseurls, baseurl);
            }
            goodsDeatilVo.setDetails(detail);
        }


        //获取规格参数
        if (StringUtils.isNotEmpty(map.get("description"))) {
            String descriptionPath = map.get("description").toString();
            //获取规格参数
            String description = ReadTxtUtil.readFile(filePath + descriptionPath);
            if (!description.contains(baseurl)) {
                description = description.replaceAll(baseurls, baseurl);
            }
            goodsDeatilVo.setDescription(description);
        }


        goodsDeatilVo.setGoodsMap(map);

        //获取图片
        IpGoodsImg ipGoodsImg = ipGoodsImgMapper.selectByGoodsid(goodId);
        if (ipGoodsImg != null) {
            goodsDeatilVo.setUrlid(ipGoodsImg.getGoodsimgid());
        }

        //获取图片path
        List<String> urlPath = ipGoodsImgMapper.selectImgPath(goodId);
        List<String> utlPathList = new ArrayList<String>();
        if (urlPath != null && urlPath.size() > 0) {
            for (String s : urlPath) {
                utlPathList.add(baseurl + s);
            }
        }
        goodsDeatilVo.setUrlpath(utlPathList);

        /**
         * 获取商品价格
         */
        goodsDeatilVo.setPriceList(this.getAllGoodsPrice(goodId));

        /**
         * 获取运费信息
         */
        goodsDeatilVo.setSuppliersList(this.getAllGoodsSupplier(goodId));


        goodsDeatilVo.setGoodsId(goodId);
        //
        return goodsDeatilVo;
    }

    /**
     * 获取商品价格
     *
     * @param goodId
     * @return
     * @throws Exception
     */
    private List<IpGoodsPriceVo> getAllGoodsPrice(Long goodId) throws Exception {
        List<IpGoodsPrice> ipGoodsPrices = ipGoodsPriceMapper.getDetailByGoodsid(goodId);
        List<IpGoodsPriceVo> ipGoodsPriceVoList = new ArrayList<IpGoodsPriceVo>();
        if (ipGoodsPrices != null && ipGoodsPrices.size() > 0) {
            IpGoodsPriceVo ipGoodsPriceVo = null;
            for (IpGoodsPrice goodsPrice : ipGoodsPrices) {
                ipGoodsPriceVo = new IpGoodsPriceVo();
                MapObjUtil.copy(goodsPrice, ipGoodsPriceVo);
                ipGoodsPriceVo.setPriceId(goodsPrice.getId().toString());
                ipGoodsPriceVoList.add(ipGoodsPriceVo);
            }
        }
        return ipGoodsPriceVoList;
    }


    /**
     * 获取商品运费
     *
     * @param goodId
     * @return
     * @throws Exception
     */
    private List<IpGoodsSuppliersVo> getAllGoodsSupplier(Long goodId) throws Exception {
        List<IpGoodsSuppliersVo> ipGoodsPrices = ipGoodsSuppliersMapper.selectByGoodsId(goodId);
        return ipGoodsPrices;
    }

    /**
     * 获取商品编码
     *
     * @return
     */
    private String getGoodSsNum() {
        String goodsNum = goodsMapper.getGoodNum();
        return goodsNum;
    }


    /**
     * 将下架成功的商品从solr移除
     *
     * @param busskey the busskey
     * @return boolean
     * @throws Exception the exception
     */
    public boolean deleteSolrData(Long busskey) throws Exception {
        String deleteDataUrl = solrDeleteData + "/" + busskey;
        boolean flag = false;
        JSONObject jsonObject = new JSONObject();
        String result = HttpXmlClient.post1(jsonObject.toString(), deleteDataUrl);
        if (StringUtils.isNotEmpty(result)) {
            if (result.equals("true")) {
                flag = true;
            }
        }
        return flag;

    }

    /**
     * 上架流程审批通过更新为上架
     *
     * @param busskey the busskey
     * @return boolean
     * @throws Exception the exception
     */
    public boolean goodUp(Long busskey) throws Exception {
        boolean flag = true;
        IpGoodsEntity ipGoodsEntity = new IpGoodsEntity();
        ipGoodsEntity.setId(busskey);
        ipGoodsEntity.setState(1);
        goodsMapper.updateGoodsStatus(ipGoodsEntity);
        //将商品添加到solr
        initSolrGoodsService = new InitSolrGoodsService();
        initSolrGoodsService.setSolrAddData(solrAddData);
        initSolrGoodsService.start(busskey, initSolrGoodsService);
        return flag;
    }

    /**
     * 下架流程审批通过更新为下架
     *
     * @param busskey the busskey
     * @return boolean
     * @throws Exception the exception
     */
    public boolean goodDown(Long busskey) throws Exception {
        boolean flag = true;
        IpGoodsEntity ipGoodsEntity = new IpGoodsEntity();
        ipGoodsEntity.setId(busskey);
        ipGoodsEntity.setState(2);
        goodsMapper.updateGoodsStatus(ipGoodsEntity);
        //将商品从solr移除
        this.deleteSolrData(busskey);
        return flag;
    }

    /**
     * 将上架的商品作废
     *
     * @param busskey the busskey
     * @return boolean
     * @throws Exception the exception
     */
    public boolean goodCancel(Long busskey) throws Exception {
        boolean flag = true;
        IpGoodsEntity ipGoodsEntity = new IpGoodsEntity();
        ipGoodsEntity.setId(busskey);
        ipGoodsEntity.setState(7);
        goodsMapper.updateGoodsStatus(ipGoodsEntity);
        return flag;
    }

    /**
     * 计算运费信息
     *
     * @param ipGoodsFreightDto
     * @return
     */
    @Override
    public float getGoodsFreight(IpGoodsFreightDto ipGoodsFreightDto) {

        float price = 0.0f;
        boolean flag = false;
        //判断区域id是否存在
        if (StringUtils.isNotEmpty(ipGoodsFreightDto.getAreaId())) {
            IpGoodsSupplierFreightVo ipGoodsSupplierFreightVo = ipGoodsSuppliersMapper.
                    getFreightPriceByAreaId(ipGoodsFreightDto);
            if (ipGoodsSupplierFreightVo != null) {
                price = Float.parseFloat(ipGoodsSupplierFreightVo.getFreightPrice().toString());
                flag = true;
            }

        }
        //计算城市
        if (!flag) {
            if (StringUtils.isNotEmpty(ipGoodsFreightDto.getCityId())) {
                IpGoodsSupplierFreightVo ipGoodsSupplierFreightVo = ipGoodsSuppliersMapper.
                        getFreightPriceByCityId(ipGoodsFreightDto);
                if (ipGoodsSupplierFreightVo != null) {
                    price = Float.parseFloat(ipGoodsSupplierFreightVo.getFreightPrice().toString());
                    flag = true;
                }

            }
        }

        //计算省
        if (!flag) {
            if (StringUtils.isNotEmpty(ipGoodsFreightDto.getProvinceId())) {
                IpGoodsSupplierFreightVo ipGoodsSupplierFreightVo = ipGoodsSuppliersMapper.
                        getFreightPriceByProvinceId(ipGoodsFreightDto);
                if (ipGoodsSupplierFreightVo != null) {
                    price = Float.parseFloat(ipGoodsSupplierFreightVo.getFreightPrice().toString());
                    flag = true;
                }

            }
        }
        return price;
    }


    /**
     * 异步修改商品的分类信息
     *
     * @return
     */
    @Override
    @Async
    public void updateGoodsCategory(GoodsCategoryDto goodsCategoryDto) throws Exception {

        String categoryId = goodsCategoryDto.getCategoryId();
        /**
         * 获取分类的供应商信息
         */
        List<IpCategorySuppliersVo> suppliersVoList = ipCategoryMapper.getSupplier(basUrl, categoryId);

        //获取该分类所有经销商
        List<String> suppliersId = new ArrayList<String>();
        //获取区域信息
        Set<String> set = new HashSet<>();
        for (IpCategorySuppliersVo ipCategorySuppliersVo : suppliersVoList) {
            suppliersId.add(ipCategorySuppliersVo.getSupplierid());
            if (StringUtils.isNotEmpty(ipCategorySuppliersVo.getSalesarea())) {
                String[] sts = ipCategorySuppliersVo.getSalesarea().split(",");
                for (String str : sts) {
                    set.add(str);
                }
            }

        }
        //修改分类对应的供应商信息
        IpCategorySuppliersVo supplier12 = ipCategoryMapper.getCategory(categoryId);
        Set<String> sortSet = new TreeSet<String>(Comparator.reverseOrder());
        sortSet.addAll(set);
        String type = supplier12.getType();
        //根据分类修改商品的厂商
        Map<String, String> map = new HashMap<>();
        if (type.equals("2")) {//三级分类
            map.put("supplier", supplier12.getSuppliersName());
            map.put("supplierid", supplier12.getSupplierid());
            map.put("supplymodename", supplier12.getSupplymodename());
            map.put("supplymodecode", supplier12.getSupplymodecode());
            map.put("thirdCategoryName", supplier12.getCategoryName());
            map.put("categoryId", categoryId);
            if (set.size() > 0) {
                map.put("salesarea", String.join(",", sortSet));
            }
            goodsMapper.updateGoodsCategory(map);
        } else {
            if (type.equals("0")) {//一级分类
                map.put("FTNPFirstCategoryName", supplier12.getCategoryName());
                map.put("categoryType", supplier12.getCategoryType());
                map.put("categoryId", categoryId);
                goodsMapper.updateFirSecCatrgoryName(map);
            } else if (type.equals("1")) {//二级分类
                map.put("FTNPSecondCategoryName", supplier12.getCategoryName());
                map.put("categoryId", categoryId);
                goodsMapper.updateFirSecCatrgoryName(map);
            }

        }

        if (type.equals("2")) {
            /**
             * 获取所有的商品信息
             */
            List<String> arraylist = goodsMapper.getGoodsId(goodsCategoryDto.getCategoryId());
            if (arraylist != null && arraylist.size() > 0) {
                for (String s : arraylist) {

                    Long goodsId = Long.parseLong(s);
                    Map<String, Object> goodsDetail = goodsMapper.getGoodsDetail(goodsId);
                    String ftnpc = null;
                    if (StringUtils.isNotEmpty(goodsDetail.get("FTNPJC"))) {
                        ftnpc = goodsDetail.get("FTNPJC").toString();
                    }
                    List<String> nowSup = new ArrayList<>();
                    List<IpGoodsSuppliersVo> goodsSuppliersVoList = this.getAllGoodsSupplier(goodsId);
                    //先判断已有的供应商是否含有修改的供应商,没有则删除
                    if (goodsSuppliersVoList != null && goodsSuppliersVoList.size() > 0) {
                        for (IpGoodsSuppliersVo ipGoodsSuppliersVo : goodsSuppliersVoList) {
                            String supplier = ipGoodsSuppliersVo.getSupplierId();
                            //判断是否存在供应商ID，存在
                            if (suppliersId.contains(supplier)) {
                                nowSup.add(supplier);
                            } else {
                                //不存在则删除
                                IpGoodsSuppliers ipGoodsSuppliers = new IpGoodsSuppliers();
                                ipGoodsSuppliers.setGoodsId(goodsId);
                                ipGoodsSuppliers.setSupplierId(supplier);
                                ipGoodsSuppliersMapper.deleteByGoodsIdAndSuppliers(ipGoodsSuppliers);
                            }
                        }
                    }
                    //判断修改供应商是否包含商品已有的
                    for (IpCategorySuppliersVo ipCategorySuppliersVo : suppliersVoList) {
                        String supplier = ipCategorySuppliersVo.getSupplierid();
                        //判断是否存在供应商ID，存在
                        if (!nowSup.contains(supplier)) {
                            IpGoodsSuppliers ipGoodsSuppliers = new IpGoodsSuppliers();
                            ipGoodsSuppliers.setGoodsId(goodsId);
                            ipGoodsSuppliers.setSupplierId(supplier);
                            ipGoodsSuppliers.setSupplierName(ipCategorySuppliersVo.getSuppliersName());
                            ipGoodsSuppliersMapper.insertSelective(ipGoodsSuppliers);
                        }
                    }
                    if (ftnpc == null || "XM".equals(ftnpc)) {
                        //同步数据到sap
                        this.initGoodsSapId("03", s, null);
                    }
                }
            }


        }

        //同步数据
        initSolrDataService.start(initSolrDataService);

    }

    /**
     * 处理商品的操作类型
     *
     * @param boGoodUpDto
     * @return
     */
    @Override
    public boolean approvalGoods(BoGoodUpDto boGoodUpDto) throws Exception {
        List<GoodsDetailDto> goods = boGoodUpDto.getGoods();
        String type = boGoodUpDto.getType();
        String createId = boGoodUpDto.getUserid();
        //判断商品是否在审批中
        Map<String, Object> goodsProcess = this.checkGoodsStatus(boGoodUpDto);
        //判断是否发起流程
        String isProcess = boGoodUpDto.getIsProcess();
        for (GoodsDetailDto good : goods) {
            String goodsname = good.getGoodsname();
            String goodsNum = good.getGoodsnum();
            String goodsid = good.getGoodsid();
            Long busskey = Long.parseLong(goodsid);
            if (isProcess.equals("Y")) {
                //根据商品id获取流程业务参数
                Map<String, Object> goodsProcessMap = (Map<String, Object>)goodsProcess.get(goodsid);
                //流程标题
                String title = ipGoodsDraftService.getTitle(goodsname, type, goodsNum);
                //流程实例号
                String instid = ipGoodsDraftService.startProcess(createId, title,goodsProcessMap);
                //保存流程记录
                ipGoodsDraftService.saveGoodProcess(good, type, createId, instid);
            } else {
                this.initGoodsStatus(busskey, type);
            }

            //保存操作记录
            this.saveGoodLog(busskey, createId, type, boGoodUpDto.getUserName());
        }

        return true;
    }

    /**
     * 校验选择的商品是否可发起上下架流程
     *
     * @param boGoodUpDto
     */
    private Map<String,Object> checkGoodsStatus(BoGoodUpDto boGoodUpDto) throws Exception {
        String userid = boGoodUpDto.getUserid();
        List<GoodsDetailDto> goods = boGoodUpDto.getGoods();
        Map<String,Object> goodsProcessMap=new HashMap<>();
        for (GoodsDetailDto good : goods) {
            String goodsname = good.getGoodsname();
            String goodsid = good.getGoodsid();
            Long goodsId = Long.parseLong(good.getGoodsid());
           List<IpProcessGoodsVo> goodsVos = goodsMapper.getGoodsNameByGoodsId(good);
            if (goodsVos.size() > 0) {
                throw new BusinessException(500, "您选择的" +
                        good.getGoodsname() + "商品正在审批中，请勿操作");
            }
            //判断商品信息是否维护

            Map<String, Object> goodsMap = goodsMapper.getGoodsDetail(goodsId);
            String state = goodsMap.get("state").toString();
            if (boGoodUpDto.getType().equals("上架")) {
                if (goodsMap == null || StringUtils.isEmpty(goodsMap.get("urlid"))) {
                    throw new BusinessException(500, "请先完善(" +
                            good.getGoodsname() + ")的基本信息！");
                }
                if ("1".equals(state)) {
                    if ("7".equals(state)) {
                        throw new BusinessException(500, "此商品(" +
                                good.getGoodsname() + ")已删除,此商品不能操作！");
                    }

                    throw new BusinessException(500, "此商品(" +
                            good.getGoodsname() + ")已上架，请勿重复操作！");
                }
            } else if (boGoodUpDto.getType().equals("下架")) {
                if ("7".equals(state)) {
                    throw new BusinessException(500, "此商品(" +
                            good.getGoodsname() + ")已删除,此商品不能操作！");
                }
                if ("2".equals(state)) {
                    throw new BusinessException(500, "此商品(" +
                            good.getGoodsname() + ")已下架，请勿重复操作！");
                }
            } else if (boGoodUpDto.getType().equals("删除")) {
                if ("1".equals(state)) {
                    throw new BusinessException(500, "此商品(" +
                            good.getGoodsname() + ")已上架，不可删除！");
                }
                if ("7".equals(state)) {
                    throw new BusinessException(500, "此商品(" +
                            good.getGoodsname() + ")已删除，请勿重复操作！");
                }
            }
            //获取流程发起的业务参数
            Map<String, Object> processBusiness = this.getProcessBusiness(goodsId, userid);

            logger.info(JSONObject.fromObject(processBusiness).toString());
            //获取商品类型
            String goodsType=processBusiness.get("GOODSTYPE").toString();
            //若为项目,判断是否检验是否有审批人
            if ("XM".equals(goodsType)){
                    this.checkGoodsXmProject(processBusiness,goodsname);
            }
            goodsProcessMap.put(goodsid,processBusiness);

        }
        return goodsProcessMap;
    }

    /**
     * 校验审批人
     * @param processBusiness
     */
    private void checkGoodsXmProject( Map<String, Object> processBusiness,String goodsName){

        if (StringUtils.isNotEmpty(processBusiness.get("SALETYPE"))){
            String saletype=processBusiness.get("SALETYPE").toString();
            //判断若为项目 则区域、城市、项目的审批人不能为空
            if (saletype.equals("F")){
                if (StringUtils.isEmpty(processBusiness.get("PROJECTCODE"))
                        || StringUtils.isEmpty(processBusiness.get("AREAPERSION"))
                        || StringUtils.isEmpty(processBusiness.get("CITYPERSIONCODE"))) {
                    throw new BusinessException(500, "此商品(" +
                            goodsName + ")未匹配到审批人员，请联系相关负责人");
                }
            }else if (saletype.equals("S")){
                //若为城市
                if (StringUtils.isEmpty(processBusiness.get("AREAPERSION"))
                        || StringUtils.isEmpty(processBusiness.get("CITYPERSIONCODE"))) {
                    throw new BusinessException(500, "此商品(" +
                            goodsName + ")未匹配到审批人员，请联系相关负责人");
                }
            }else {

                if (StringUtils.isEmpty(processBusiness.get("AREAPERSION"))) {
                    throw new BusinessException(500, "此商品(" +
                            goodsName + ")未匹配到审批人员，请联系相关负责人");
                }
            }
        }else{
                throw  new BusinessException(500,"请维护商品("+goodsName+")的销售价格");

        }
    }

    /**
     * 操作类型
     *
     * @param busskey
     * @param type
     */
    private void initGoodsStatus(Long busskey, String type) throws Exception {

        //修改商品的分类信息隐藏
        Map<String, Object> map = goodsMapper.getGoodsDetail(busskey);
        String categoryId = map.get("FTNPThreeCategoryCode").toString();
        String ftnpjc = null;
        if (StringUtils.isNotEmpty(map.get("FTNPJC"))) {
            ftnpjc = map.get("FTNPJC").toString();
        }

        if (type.equals("上架") || type.equals("新增") || type.equals("编辑")) {
            this.goodUp(busskey);
            this.updateGoodsCategory(categoryId);
            if (ftnpjc == null || "XM".equals(ftnpjc)) {
                //同步sap
                this.initGoodsSapId("03", busskey.toString(), null);
            }

        } else if (type.equals("下架")) {
            this.goodDown(busskey);
            this.updateGoodsCateHide(categoryId);
            if (ftnpjc == null || "XM".equals(ftnpjc)) {
                //同步sap1
                this.initGoodsSapId("03", busskey.toString(), "X");
            }

        } else if (type.equals("删除")) {
            this.goodCancel(busskey);
            this.updateGoodsCateHide(categoryId);
        }


    }

    /**
     * 保存商品的操作记录
     *
     * @param busskey
     * @param userId
     * @param type
     */
    private void saveGoodLog(Long busskey, String userId, String type, String userName) {
        Map<String, Object> map = goodsMapper.getGoodsDetail(busskey);
        IpGoodsLogsWithBLOBs logsWithBLOBs = new IpGoodsLogsWithBLOBs();
        logsWithBLOBs.setGoodsid(Integer.parseInt(busskey.toString()));
        logsWithBLOBs.setOperatetype(type);
        logsWithBLOBs.setOperatedate(DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
        logsWithBLOBs.setSubmitpeople(userId);
        logsWithBLOBs.setPrice(map.get("price").toString());
        logsWithBLOBs.setSuccessdate(map.get("successdate").toString());
        logsWithBLOBs.setSubmitpeoplename(userName);
        ipGoodsLogsMapper.insertSelective(logsWithBLOBs);
    }

    /**
     * 流程审批结束
     */
    @Override
    public void initGoodsProcess(IpGoodsProcessDto ipGoodsProcessDto) throws Exception {
        String instid = ipGoodsProcessDto.getInstid();
        if (ipGoodsProcessDto.getProcessStatus().equals("PASS")) {
            this.initPass(instid);
        }
        /**
         * 更新合同状态
         */
        IpGoodProcess ipGoodProcess = new IpGoodProcess();
        ipGoodProcess.setInstid(instid);
        ipGoodProcess.setStatus(ipGoodsProcessDto.getProcessStatus());
        ipGoodProcessMapper.updteStatusEnd(ipGoodProcess);
    }

    /**
     * 流程审批通过
     *
     * @param instid
     * @throws Exception
     */
    private void initPass(String instid) throws Exception {
        //先获取数据
        IpGoodProcess ipGoodProcess = ipGoodsDraftService.getGoodProcess(instid);
        String type = ipGoodProcess.getType();
        int status = 6;
        if (type.equals("上架") || type.equals("新增") || type.equals("编辑")) {
            status = 1;
        } else if (type.equals("下架")) {
            status = 2;
        } else if (type.equals("删除")) {
            status = 7;
        }

        GoodsGetDetailDto goodsGetDetailDto = new GoodsGetDetailDto();
        goodsGetDetailDto.setInstid(instid);
        GoodsDeatilVo getGoodsDetail = this.getGoodsDetail(goodsGetDetailDto);
        Map<String, Object> goodsMap = getGoodsDetail.getGoodsMap();
        String categoryId = goodsMap.get("FTNPThreeCategoryCode").toString();

        if (status == 1) {
            this.updateGoodsCategory(categoryId);
        } else {
            this.updateGoodsCateHide(categoryId);
        }
        //修改主表数据
        Long goodsId = ipGoodProcess.getGoodsid();
        this.updateGoodMain(getGoodsDetail, status, goodsId);

        //修改供应1商数据运费
        this.saveOrUpdateSupplier(goodsId, getGoodsDetail);

        //修改价格

        this.updateGoodsPrice(goodsId, getGoodsDetail);

        //修改图片

        this.saveOrUpdateFileImg(goodsId, getGoodsDetail);

        /**
         * 更新商品的状态
         */
        this.initGoodsStatus(goodsId, type);
        String ftnpjc = null;
        if (StringUtils.isNotEmpty(goodsMap.get("FTNPThreeCategoryCode"))) {
            ftnpjc = goodsMap.get("FTNPThreeCategoryCode").toString();
        }

        if (ftnpjc == null || "XM".equals(ftnpjc)) {
            if (status == 1) {
                this.initGoodsSapId("03", goodsId.toString(), null);
            } else if (status == 2) {
                this.initGoodsSapId("03", goodsId.toString(), "X");
            }
        }
    }

    /**
     * 修改主表信息
     *
     * @param getGoodsDetail
     * @param status
     * @param busskey
     */
    private void updateGoodMain(GoodsDeatilVo getGoodsDetail, int status, Long busskey) {
        //先保存主表信息
        InsertMap insertGoodsMap = new InsertMap();
        insertGoodsMap.setTableName("ip_s_goods");
        Map<String, Object> goodsMap = getGoodsDetail.getGoodsMap();
        goodsMap.put("state", status);
        goodsMap.put("id", busskey);
        if (goodsMap.containsKey("userid")){
            goodsMap.remove("userid");
        }

        goodsMap.put("shelvesdate", DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
        insertGoodsMap.setParams(goodsMap);

        insertGoodsMap.setId(busskey);
        goodsMapper.updateGoods(insertGoodsMap);
    }


    /**
     * 保存供应商运费
     */
    private void saveOrUpdateSupplier(Long busskey, GoodsDeatilVo getGoodsDetail) throws Exception {


        supplierFreightMapper.deleteByGoodsId(busskey);
        ipGoodsSuppliersMapper.deleteByGoodsId(busskey);
        //获取供应商信息
        List<IpGoodsSuppliersVo> suppliersDtoListList = getGoodsDetail.getSuppliersList();
        if (suppliersDtoListList != null && suppliersDtoListList.size() > 0) {
            IpGoodsSuppliers ipGoodsSuppliers = null;
            for (IpGoodsSuppliersVo ipGoodsSuppliersVo : suppliersDtoListList) {
                ipGoodsSuppliers = new IpGoodsSuppliers();
                MapObjUtil.copy(ipGoodsSuppliersVo, ipGoodsSuppliers);
                ipGoodsSuppliers.setGoodsId(busskey);
                ipGoodsSuppliersMapper.insertSelective(ipGoodsSuppliers);
                Long suppliersId = ipGoodsSuppliers.getId();
                //获取供应商运费信息
                List<IpGoodsSupplierFreightVo> ipGoodsPriceDtos = ipGoodsSuppliersVo.getSupplierFreightList();
                if (ipGoodsPriceDtos != null && ipGoodsPriceDtos.size() > 0) {
                    IpGoodsSupplierFreight ipGoodsSupplierFreight = null;

                    for (IpGoodsSupplierFreightVo ipGoodsSupplierFreightDto : ipGoodsPriceDtos) {
                        ipGoodsSupplierFreight = new IpGoodsSupplierFreight();
                        MapObjUtil.copy(ipGoodsSupplierFreightDto, ipGoodsSupplierFreight);
                        ipGoodsSupplierFreight.setGoodsId(busskey);
                        ipGoodsSupplierFreight.setGoodsSuppliersId(suppliersId);
                        supplierFreightMapper.insertSelective(ipGoodsSupplierFreight);
                    }
                }
            }
        }
    }

    /**
     * 保存商品图片
     *
     * @param busskey
     * @param getGoodsDetail
     */
    private void saveOrUpdateFileImg(Long busskey, GoodsDeatilVo getGoodsDetail) {
        ipGoodsImgMapper.deleteByGoodsId(busskey);
        if (StringUtils.isNotEmpty(getGoodsDetail.getUrlid())) {
            IpGoodsImg ipGoodsImg = new IpGoodsImg();
            ipGoodsImg.setGoodsid(busskey);
            ipGoodsImg.setGoodsimgid(getGoodsDetail.getUrlid());
            ipGoodsImgMapper.insertSelective(ipGoodsImg);
        }
    }

    /**
     * 保存商品价格数据
     *
     * @param busskey
     * @param getGoodsDetail
     */
    private void updateGoodsPrice(Long busskey, GoodsDeatilVo getGoodsDetail) throws Exception {
        ipGoodsPriceMapper.updateByGoodsid(busskey);
        List<IpGoodsPriceVo> ipGoodsPriceDtos = getGoodsDetail.getPriceList();
        if (ipGoodsPriceDtos != null && ipGoodsPriceDtos.size() > 0) {
            IpGoodsPrice goodsPrice = null;

            for (IpGoodsPriceVo ipGoodsPriceDto : ipGoodsPriceDtos) {
                goodsPrice = new IpGoodsPrice();
                String priceId = ipGoodsPriceDto.getPriceId();
                MapObjUtil.copy(ipGoodsPriceDto, goodsPrice);
                goodsPrice.setGoodsId(busskey);
                goodsPrice.setIsDelete("N");
                int detailByGoodsidCount = 0;
                if (StringUtils.isNotEmpty(priceId)) {
                    Long id = Long.parseLong(priceId);
                    detailByGoodsidCount = ipGoodsPriceMapper.getDetailByGoodsidCount(id, busskey);
                }


                if (detailByGoodsidCount > 0) {
                    ipGoodsPriceMapper.updateByPrimaryKeySelective(goodsPrice);
                } else {
                    goodsPrice.setId(null);
                    ipGoodsPriceMapper.insertSelective(goodsPrice);
                }

            }
        }
    }

    /**
     * 修改分类显示
     *
     * @param categoryId
     */
    private void updateGoodsCategory(String categoryId) {
        ipCategoryMapper.updateCategory(basUrl, categoryId, "1");
    }

    /***
     * 修改分类不显示
     * @param categoryId
     */
    private void updateGoodsCateHide(String categoryId) {
        int num = goodsMapper.getGoodsIdCount(categoryId);
        if (num == 0) {
            ipCategoryMapper.updateCategory(basUrl, categoryId, "0");
        }
    }

    /***
     * 流程审批不通过
     */
    private void initiFail() {

    }

    /**
     * 获取操作记录
     *
     * @return
     */
    @Override
    public PageInfo getGoodsLogs(GoodsGetDetailDto goodsGetDetailDto) {
        PageHelper.startPage(goodsGetDetailDto.getPagenum(), goodsGetDetailDto.getPagesize());
        List<IpGoodsLogsVo> logList = ipGoodsLogsMapper.getLogByGoodsId(goodsGetDetailDto.getGoodsId());
        PageInfo pageInfo = new PageInfo(logList);
        return pageInfo;
    }


    /**
     * 计算 不是实时价商品的价格
     * 1.获取商品ID
     * 2,先根据登录用户是内部还是外部
     * 4.根据客户类型获取商品价格
     * 5.如果未维护则直接返回商品的标准价格
     * @return
     */
    /**
     * 计算商品的价格
     *
     * @param priceDto
     * @return
     */
    @Override
    public IpGoodPriceVo getGoodsPrice(IpGoodPriceDto priceDto) throws Exception {
        IpGoodPriceVo ipGoodPriceVo = null;
        String userId = priceDto.getUserid();
        Map<String, Object> goodsMap = goodsMapper.getGoodsDetail(priceDto.getGoodsId());
        if (StringUtils.isNotEmpty(priceDto.getIsSupplier()) && priceDto.getIsSupplier().equals("Y") && StringUtils.isEmpty(priceDto.getIsSupplierPrice())) {
            ipGoodPriceVo = new IpGoodPriceVo();
            ipGoodPriceVo.setTaxPrice(goodsMap.get("allPrice").toString());
            ipGoodPriceVo.setNotaxPrice(goodsMap.get("allOutPrice").toString());
        } else {
            if (priceDto.getIsOutUser().equals("Y")) {
                ipGoodPriceVo = this.goodsPriceOut(priceDto);
            } else {
                ipGoodPriceVo = this.goodsPriceInside(priceDto);
            }
            if (ipGoodPriceVo == null || ipGoodPriceVo.getTaxPrice() == null) {
                ipGoodPriceVo = new IpGoodPriceVo();
                ipGoodPriceVo.setTaxPrice("暂无报价");
            }
        }


        return ipGoodPriceVo;
    }

    /**
     * 1.先根据当前选择的公司查询是否有费用‘
     * 2.如果 查询不到 则向上找上一
     * 获取内部用户商品费用
     *
     * @param priceDto
     * @return
     */
    private IpGoodPriceVo goodsPriceInside(IpGoodPriceDto priceDto) {
        String userId = priceDto.getUserid();
        String companyId = priceDto.getCompanyId();
        Long goodsId = priceDto.getGoodsId();
        IpGoodPriceVo ipGoodPriceVo = null;
        //查上级，上级是否存在价格
        String string = ipGoodsPriceMapper.getParentTree(basUrl, companyId);
        String[] str = string.split(",");
        if (str != null) {
            String parentId = null;
            for (int i = str.length - 1; i >= 0; i--) {
                companyId = str[i];
                if (i > 0) {
                    parentId = str[i - 1];
                }
                ipGoodPriceVo = this.getInsidePrice(companyId, goodsId, parentId);
                //判断是否取到价格，若取到则跳出
                if (ipGoodPriceVo != null && ipGoodPriceVo.getTaxPrice() != null) {
                    break;
                } else {
                    ipGoodPriceVo = null;
                }
            }
        }
        return ipGoodPriceVo;
    }

    /**
     * 根据商品ID获取费用
     *
     * @param companyId
     * @param goodsId
     * @return
     */
    private IpGoodPriceVo getInsidePrice(String companyId, Long goodsId, String parentId) {
        IpGoodPriceVo ipGoodPriceVo = new IpGoodPriceVo();
        IpGoodsPrice ipGoodsPrice = ipGoodsPriceMapper.
                getInsideGoodsPriceByCompanyId(companyId, goodsId, parentId);
        if (ipGoodsPrice != null) {
            ipGoodPriceVo = new IpGoodPriceVo();
            ipGoodPriceVo.setTaxPrice(ipGoodsPrice.getTaxPrice().toString());
            ipGoodPriceVo.setNotaxPrice(ipGoodsPrice.getNotaxPrice().toString());
        }
        return ipGoodPriceVo;
    }

    /**
     * 计算外部用户登录商品费用
     *
     * @param priceDto
     * @return
     */
    private IpGoodPriceVo goodsPriceOut(IpGoodPriceDto priceDto) throws Exception {
        IpGoodPriceVo ipGoodPriceVo = null;
        String customerType = priceDto.getCustomerType();
        Long goodsId = priceDto.getGoodsId();
        if (StringUtils.isNotEmpty(customerType) && "GR".equals(customerType)) {
            String userid = priceDto.getUserid();
            List<LoginUserVo> userDetail = ipGoodsPriceMapper.getUserDetail(userid);
            if (userDetail != null && userDetail.size() > 0) {
                LoginUserVo loginUserVo = userDetail.get(0);
                //获取员工类型
                String remark = loginUserVo.getCustType();
                if ("2".equals(remark)) {
                    String attrid = priceDto.getAttrid();
                    ipGoodPriceVo = this.getCustNbygPrice(goodsId, attrid);
                }
            }
        } else {
            IpGoodsPrice ipGoodsPrice = ipGoodsPriceMapper.getOutGoodsPrice(priceDto);
            if (ipGoodsPrice != null) {
                ipGoodPriceVo = new IpGoodPriceVo();
                ipGoodPriceVo.setTaxPrice(ipGoodsPrice.getTaxPrice().toString());
                ipGoodPriceVo.setNotaxPrice(ipGoodsPrice.getNotaxPrice().toString());
            }
        }

        return ipGoodPriceVo;
    }

    /**
     * 内部员工获取商品价格
     *
     * @Param: * @param goodsId
     * @return: com.ekingwin.bas.cloud.vo.IpGoodPriceVo
     * @Author: zhangjie
     * @Date: 2019/10/28 20:56
     */
    private IpGoodPriceVo getCustNbygPrice(Long goodsId, String attrid) throws Exception {
        IpGoodPriceVo ipGoodPriceVo = null;
        IpGoodsPrice outGoodsPriceNbyg = null;
        if (StringUtils.isNotEmpty(attrid)) {
            long parseLong = Long.parseLong(attrid);
            outGoodsPriceNbyg = ipGoodsPriceMapper.selectByPrimaryKey(parseLong);
        } else {
            outGoodsPriceNbyg = ipGoodsPriceMapper.getOutGoodsPriceNbyg(goodsId);
        }

        if (outGoodsPriceNbyg != null) {
            ipGoodPriceVo = new IpGoodPriceVo();
            this.commonBuild(ipGoodPriceVo, outGoodsPriceNbyg);
            ipGoodPriceVo.setModelGoodsId(outGoodsPriceNbyg.getId().toString());
            /*ipGoodPriceVo.setTaxPrice(outGoodsPriceNbyg.getTaxPrice().toString());
            ipGoodPriceVo.setNotaxPrice(outGoodsPriceNbyg.getNotaxPrice().toString());*/
        } else {
            List<IpGoodsPrice> outGoodsPriceList = ipGoodsPriceMapper.getOutGoodsPriceList(goodsId);
            if (outGoodsPriceList != null && outGoodsPriceList.size() > 0) {
                outGoodsPriceNbyg = outGoodsPriceList.get(0);
                ipGoodPriceVo = new IpGoodPriceVo();
                this.commonBuild(ipGoodPriceVo, outGoodsPriceNbyg);
                ipGoodPriceVo.setModelGoodsId(outGoodsPriceNbyg.getId().toString());
            }

        }

        return ipGoodPriceVo;
    }

    /**
     * 导出excel
     */
    @Override
    public void exportExcel(String goodsId, HttpServletResponse response, String goodsName) throws Exception {
        ExportExcelUtil<IpGoodsExcelVo> util = new ExportExcelUtil<IpGoodsExcelVo>();


        List<IpGoodsExcelVo> goodsExcelVos = ipGoodsPriceMapper.getGoodsPriceByGoodsId(goodsId);
        // excel 表文件名
        String fileName = String.valueOf(System.currentTimeMillis()).substring(4, 13) + ".xlsx";
        String fileName11 = URLEncoder.encode(fileName, "UTF-8");
        String headStr = "attachment; filename=\"" + fileName11 + "\"";
        response.setContentType("APPLICATION/OCTET-STREAM");
        response.setHeader("Content-Disposition", headStr);
        OutputStream out = response.getOutputStream();
        String[] goodCloumn = {"商品名称", "商品编码", "品牌"
                , "规格型号", "单位", "含税原价", "税率", "不含税原价",
                "客户类型", "公司名称", "定价类型"
                , "调价额度", "最终不含税价", "最终含税价"};
        util.exportExcel(goodsName, goodCloumn, goodsExcelVos, out, ExportExcelUtil.EXCEl_FILE_2007);
        out.flush();
        out.close();
    }

    /**
     * 获取商品描述
     *
     * @param goodsId
     * @param goodsNum
     * @return
     */
    private IpGoodsDetailVo getGoodsDetailById(Long goodsId, String goodsNum) throws Exception {
        IpGoodsDetailVo ipGoodsDetailVo = new IpGoodsDetailVo();
        Map<String, Object> map = this.getGoodsMap(goodsId, goodsNum);
        String detailPath = map.get("details").toString();
        //获取商品1详情
        String detail = ReadTxtUtil.readFile1(filePath + detailPath);
        ipGoodsDetailVo.setDetail(detail);
        return ipGoodsDetailVo;
    }

    /**
     * 获取商品描述详情
     *
     * @param goodsGetDetailDto
     * @return
     */
    @Override
    public IpGoodsDetailVo getGoodsDetails(GoodsGetDetailDto goodsGetDetailDto) throws Exception {
        IpGoodsDetailVo ipGoodsDetailVo = null;
        //获取商品详情
        if (StringUtils.isNotEmpty(goodsGetDetailDto.getGoodsId())) {
            ipGoodsDetailVo = this.getGoodsDetailById(goodsGetDetailDto.getGoodsId(), null);
        } else {
            String instid = goodsGetDetailDto.getInstid();
            ipGoodsDetailVo = ipGoodsDraftService.getGoodsDetailByInstid(instid);
            if (ipGoodsDetailVo == null) {
                IpGoodProcess ipGoodProcess = ipGoodsDraftService.getGoodProcess(instid);
                ipGoodsDetailVo = this.getGoodsDetailById(ipGoodProcess.getGoodsid(), ipGoodProcess.getGoodsnum());
            }

        }
        return ipGoodsDetailVo;
    }

    /**
     * 商品导入
     *
     * @param file
     * @throws Exception
     */
    @Override
    public GoodsImportVo importGoods(MultipartFile file, String userId, String categoryType) throws Exception {
        ImportExcelUtil util = new ImportExcelUtil();
        InputStream input = null;
        List<List<Object>> lists = null;
        if (file.isEmpty()) {
            throw new BusinessException(500, "导入文件为空，请先添加Excel文件！");
        }
        int success = 0;
        int fail = 0;
        int total = 0;
        String fileName = file.getOriginalFilename();
        input = file.getInputStream();
        lists = util.getBankListByExcel(input, fileName);
        input.close();
        List<String> collun = new ArrayList<String>();
        Set<String> arraySet = new HashSet<String>();

        List<Map<String, Object>> arrayList = new ArrayList<>();
        //分类信息
        Map<String, Object> categoryMapCode = new HashMap<>();
        //存储分类的销售区域，和授权商
        Map<String, Object> categorySupplier = new HashMap<>();
        for (int i = 0; i < lists.size(); i++) {
            Map<String, Object> map = new HashMap<>();
            List<Object> objects = lists.get(i);
            if (i == 0) {
                for (int j = 0; j < objects.size(); j++) {
                    Object o = objects.get(j);
                    if (o != null && StringUtils.isNotEmpty(o)) {
                        collun.add(util.getFormat(String.valueOf(o)));
                    } else {
                        break;
                    }

                }
            } else if (i > 1) {

                for (int j = 0; j < objects.size(); j++) {
                    Object o = objects.get(j);
                    if (j < collun.size()) {
                        map.put(collun.get(j), util.getFormat(String.valueOf(o)));
                    }

                }
                //导入数据处理
                if (StringUtils.isNotEmpty(map.get("name"))) {
                    total++;
                    if (StringUtils.isNotEmpty(map.get("pricetype"))) {
                        String priceType = map.get("pricetype").toString();
                        if (priceType.equals("实时价")) {
                            map.put("pricetype", "0");
                        } else {
                            map.put("pricetype", "1");
                        }
                    }
                    //判断三级分类能否取到
                    if (StringUtils.isNotEmpty(map.get("FTNPFirstCategoryCode"))
                            && StringUtils.isNotEmpty(map.get("FTNPSecondCategoryCode")) &&
                            StringUtils.isNotEmpty(map.get("FTNPThreeCategoryCode"))) {
                        map.put("baseUrl", basUrl);
                        String threeCategoryCode = map.get("FTNPThreeCategoryCode").toString();
                        Map<String, Object> mapCate = null;
                        if (!categoryMapCode.containsKey(threeCategoryCode)) {
                            //获取分类信息
                            if ("GR".equals(categoryType)) {
                                mapCate = ipCategoryMapper.getGoodCateCodeByGr(map);
                            } else {

                                mapCate = ipCategoryMapper.getGoodCateCode(map);
                            }
                            categoryMapCode.put(threeCategoryCode, mapCate);
                        } else {
                            mapCate = (Map<String, Object>) categoryMapCode.get(threeCategoryCode);
                        }

                        if (mapCate != null) {
                            success++;
                            map.remove("baseUrl");
                            String goodsCategory = null;
                            if (StringUtils.isNotEmpty(mapCate.get("categoryType"))) {
                                goodsCategory = mapCate.get("categoryType").toString();
                                mapCate.remove("categoryType");
                            }
                            //设置商品类型
                            map.put("FTNPJC", goodsCategory);

                            String threcode = mapCate.get("FTNPThreeCategoryCode").toString();
                            List<IpCategorySuppliersVo> suppliersVoList = null;
                            if (!categorySupplier.containsKey(threcode)) {
                                suppliersVoList = ipCategoryMapper.getSupplier(basUrl, threcode);
                                categorySupplier.put(threcode, suppliersVoList);
                            }
                            map.putAll(mapCate);
                            arrayList.add(map);
                        } else {
                            arraySet.add(map.get("name").toString());
                            fail++;
                        }
                    } else {
                        arraySet.add(map.get("name").toString());
                        fail++;
                    }

                }
            }

        }
        if (arrayList != null && arrayList.size() > 0) {
            this.saveGoodList(arrayList, categorySupplier, userId);
        }
        GoodsImportVo goodsImportVo = new GoodsImportVo();
        goodsImportVo.setFail(fail);
        goodsImportVo.setSuccess(success);
        goodsImportVo.setTotal(total);
        if (arraySet != null && arraySet.size() > 0) {
            String strName = String.join(",", arraySet);
            goodsImportVo.setFailName(strName);
        }

        return goodsImportVo;
    }

    /*
     public GoodsImportVo importGoodsGr(MultipartFile file, String userId, String categoryType) throws Exception {
         ImportExcelUtil util = new ImportExcelUtil();
         InputStream input = null;
         List<List<Object>> lists = null;
         if (file.isEmpty()) {
             throw new BusinessException(500, "导入文件为空，请先添加Excel文件！");
         }
         int success = 0;
         int fail = 0;
         int total = 0;
         String fileName = file.getOriginalFilename();
         input = file.getInputStream();
         lists = util.getBankListByExcel(input, fileName);
         input.close();
         List<String> collun = new ArrayList<String>();
         Set<String> arraySet = new HashSet<String>();

         List<Map<String, Object>> arrayList = new ArrayList<>();
         //分类信息
         Map<String, Object> categoryMapCode = new HashMap<>();
         for (int i = 0; i < lists.size(); i++) {
             Map<String, Object> map = new HashMap<>();
             List<Object> objects = lists.get(i);
             if (i == 0) {
                 for (int j = 0; j < objects.size(); j++) {
                     Object o = objects.get(j);
                     if (o != null && StringUtils.isNotEmpty(o)) {
                         collun.add(util.getFormat(String.valueOf(o)));
                     } else {
                         break;
                     }

                 }
             } else if (i > 1) {

                 for (int j = 0; j < objects.size(); j++) {
                     Object o = objects.get(j);
                     if (j < collun.size()) {
                         map.put(collun.get(j), util.getFormat(String.valueOf(o)));
                     }

                 }
                 //导入数据处理
                 if (StringUtils.isNotEmpty(map.get("name"))) {
                     total++;
                     if (StringUtils.isNotEmpty(map.get("pricetype"))) {
                         String priceType = map.get("pricetype").toString();
                         if (priceType.equals("实时价")) {
                             map.put("pricetype", "0");
                         } else {
                             map.put("pricetype", "1");
                         }
                     }
                     //判断三级分类能否取到
                     if (StringUtils.isNotEmpty(map.get("FTNPFirstCategoryCode"))
                             && StringUtils.isNotEmpty(map.get("FTNPSecondCategoryCode")) &&
                             StringUtils.isNotEmpty(map.get("FTNPThreeCategoryCode"))) {
                         map.put("baseUrl", basUrl);
                         String threeCategoryCode = map.get("FTNPThreeCategoryCode").toString();
                         Map<String, Object> mapCate = null;
                         if (!categoryMapCode.containsKey(threeCategoryCode)) {
                             //获取分类信息
                             mapCate = ipCategoryMapper.getGoodCateCodeByGr(map);
                             categoryMapCode.put(threeCategoryCode, mapCate);
                         } else {
                             mapCate = (Map<String, Object>) categoryMapCode.get(threeCategoryCode);
                         }

                         if (mapCate != null) {
                             success++;
                             map.remove("baseUrl");
                             String goodsCategory=null;
                             if (StringUtils.isNotEmpty(mapCate.get("categoryType"))){
                                 goodsCategory =mapCate.get("categoryType").toString();
                                 mapCate.remove("categoryType");
                             }
                             //设置商品类型
                             map.put("FTNPJC",goodsCategory);
                             map.putAll(mapCate);
                             arrayList.add(map);
                         } else {
                             arraySet.add(map.get("name").toString());
                             fail++;
                         }
                     } else {
                         arraySet.add(map.get("name").toString());
                         fail++;
                     }

                 }
             }

         }
         if (arrayList != null && arrayList.size() > 0) {
             this.saveGoodList(arrayList, null, userId);
         }
         GoodsImportVo goodsImportVo = new GoodsImportVo();
         goodsImportVo.setFail(fail);
         goodsImportVo.setSuccess(success);
         goodsImportVo.setTotal(total);
         if (arraySet != null && arraySet.size() > 0) {
             String strName = String.join(",", arraySet);
             goodsImportVo.setFailName(strName);
         }

         return goodsImportVo;
     }*/
    private void saveGoodList(List<Map<String, Object>> arrayList,
                              Map<String, Object> categorySupplier, String userid) {
        for (Map<String, Object> map : arrayList) {
            map.put("state", "6");
            if (StringUtils.isNotEmpty(map.get("stock"))) {
                map.put("stock", map.get("stock"));
            } else {
                map.put("stock", "9999999999999999");
            }

            map.put("userid", userid);
            map.put("goodsnum", this.getGoodSsNum());
            map.put("price", map.get("allPrice"));
            String threcode = map.get("FTNPThreeCategoryCode").toString();
            List<IpCategorySuppliersVo> suppliersVoList = (List<IpCategorySuppliersVo>) categorySupplier.get(threcode);
            if (suppliersVoList != null && suppliersVoList.size() > 0) {
                Set<String> area = this.getGoodsSalesarea(suppliersVoList);
                map.put("salesarea", String.join(",", area));

            }
            //保存基本信息
            Long goodsId = this.saveGood(map);
            //保存商品供应商
            this.saveSupplier(suppliersVoList, goodsId);

        }
    }

    /**
     * 保存商品信息
     *
     * @param goodsMap
     * @return
     */
    private Long saveGood(Map<String, Object> goodsMap) {
        //先保存主表信息
        InsertMap insertGoodsMap = new InsertMap();
        insertGoodsMap.setTableName(IpGoodsEntity.TABLENAME);
        insertGoodsMap.setParams(goodsMap);
        goodsMap.put("inputdate", DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
        goodsMapper.insertGoods(insertGoodsMap);
        return insertGoodsMap.getId();
    }

    private void saveSupplier(List<IpCategorySuppliersVo> suppliersVoList, Long busskey) {
        //判断修改供应商是否包含商品已有的
        for (IpCategorySuppliersVo ipCategorySuppliersVo : suppliersVoList) {

            IpGoodsSuppliers ipGoodsSuppliers = new IpGoodsSuppliers();
            ipGoodsSuppliers.setGoodsId(busskey);
            ipGoodsSuppliers.setSupplierId(ipCategorySuppliersVo.getSupplierid());
            ipGoodsSuppliers.setSupplierName(ipCategorySuppliersVo.getSuppliersName());
            ipGoodsSuppliersMapper.insertSelective(ipGoodsSuppliers);
        }
    }


    /**
     * 获取销售区域
     *
     * @param suppliersVoList
     * @return
     */
    private Set<String> getGoodsSalesarea(List<IpCategorySuppliersVo> suppliersVoList) {
        //获取区域信息
        Set<String> set = new HashSet<>();
        for (IpCategorySuppliersVo ipCategorySuppliersVo : suppliersVoList) {
            if (StringUtils.isNotEmpty(ipCategorySuppliersVo.getSalesarea())) {
                String[] sts = ipCategorySuppliersVo.getSalesarea().split(",");
                for (String str : sts) {
                    set.add(str);
                }
            }
        }
        Set<String> sortSet = new TreeSet<String>(Comparator.reverseOrder());
        sortSet.addAll(set);
        return sortSet;
    }

    /**
     * 商品库
     *
     * @Param: * @param GoodsMessageVo
     * @return: com.ekingwin.bas.cloud.vo.GoodsMessageVo
     * @Author: zhangjie
     * @Date: 2019/10/21 21:58
     */
    @Override
    public GoodsMessageVo getGoodsMessage(GoodsMessageCategoryDto messageCategoryDto) {
        messageCategoryDto.setImgUrl(baseurl);
        GoodsMessageVo goodsMessageVo = new GoodsMessageVo();
        //先获取所有的状态
        List<GoodsMessageStatusVo> goodsStatus = goodsMapper.getGoodsStatus(messageCategoryDto);
        goodsMessageVo.setState(goodsStatus);
        //获取一级分类
        List<GoodsMessageCategoryVo> firstCategory = goodsMapper.getFirstCategory(messageCategoryDto);
        goodsMessageVo.setFTNPFirstCategory(firstCategory);

        //获取二级分类
        List<GoodsMessageCategoryVo> secondCategory = goodsMapper.getSecondCategory(messageCategoryDto);
        goodsMessageVo.setFTNPSecondCategory(secondCategory);

        //获取三级分类
        List<GoodsMessageCategoryVo> threadCategory = goodsMapper.getThreadCategory(messageCategoryDto);
        goodsMessageVo.setFTNPThreeCategory(threadCategory);
        //获取商品
        PageHelper.startPage(messageCategoryDto.getPagenum(), messageCategoryDto.getPagesize());
        List<Map<String, Object>> allGoodList = goodsMapper.getAllGoodList(messageCategoryDto);
        String categoryType = messageCategoryDto.getCategoryType();
        if (StringUtils.isNotEmpty(categoryType) && "GR".equals(categoryType)) {
            if (allGoodList != null && allGoodList.size() > 0) {
                for (Map<String, Object> stringObjectMap : allGoodList) {
                    Long id = Long.parseLong(stringObjectMap.get("id").toString());
                    IpGoodsPrice outGoodsPriceNbyg = ipGoodsPriceMapper.getOutGoodsPriceNbyg(id);
                    if (outGoodsPriceNbyg != null) {
                        stringObjectMap.put("price", outGoodsPriceNbyg.getAllPrice());
                    }
                }
            }
        }
        PageInfo<Map<String, Object>> pageInfo = new PageInfo<Map<String, Object>>(allGoodList);
        goodsMessageVo.setPageCount(pageInfo.getTotal());
        List<Map<String, Object>> list = pageInfo.getList();
        goodsMessageVo.setGoodsList(list);
        return goodsMessageVo;
    }

    /**
     * 商品复制
     *
     * @Param: * @param goodsId
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/22 19:44
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void copyGoods(CopyGoodsDto copyGoodsDto) throws Exception {
        GoodsDeatilVo goodsDeatilVo = this.getGoodsDetailByGoodsId(copyGoodsDto.getGoodsId(), null);

        GoodsAddDto goodsAddDto = new GoodsAddDto();
        goodsAddDto.setGoodsId(null);

        //商品信息
        Map<String, Object> goodsMap = goodsDeatilVo.getGoodsMap();
        goodsMap.put("id", null);
        goodsMap.put("goodsnum", "");
        goodsAddDto.setGoodsMap(goodsMap);
        goodsAddDto.setUrlid(goodsMap.get("urlid") + "");
        goodsAddDto.setDescription(goodsDeatilVo.getDescription());
        goodsAddDto.setDetails(goodsDeatilVo.getDetails());
        goodsAddDto.setDirectiveground("N");
        goodsAddDto.setStatus(0);
        goodsAddDto.setType("新增");
        goodsAddDto.setUserid(copyGoodsDto.getUserid());
        goodsAddDto.setUserName(copyGoodsDto.getUserName());
        //供应商价格
        List<IpGoodsPriceVo> priceList = goodsDeatilVo.getPriceList();
        List<IpGoodsPriceDto> priceDtos = null;
        if (priceList != null && priceList.size() > 0) {
            priceDtos = new ArrayList<IpGoodsPriceDto>();
            IpGoodsPriceDto ipGoodsPriceDto = null;
            for (IpGoodsPriceVo priceVo : priceList) {
                ipGoodsPriceDto = new IpGoodsPriceDto();
                MapObjUtil.copy(priceVo, ipGoodsPriceDto);
                ipGoodsPriceDto.setId(null);
                priceDtos.add(ipGoodsPriceDto);
            }
            goodsAddDto.setPriceDtoList(priceDtos);
        }
        //供应商运费
        List<IpGoodsSuppliersDto> suppliersDtoList = null;
        List<IpGoodsSuppliersVo> suppliersList = goodsDeatilVo.getSuppliersList();
        if (suppliersList != null && suppliersList.size() > 0) {
            suppliersDtoList = new ArrayList<>();
            IpGoodsSuppliersDto ipGoodsSuppliersDto = null;
            List<IpGoodsSupplierFreightDto> supplierFreightListDto = null;
            for (IpGoodsSuppliersVo ipGoodsSuppliersVo : suppliersList) {
                ipGoodsSuppliersDto = new IpGoodsSuppliersDto();
                MapObjUtil.copy(ipGoodsSuppliersVo, ipGoodsSuppliersDto);
                List<IpGoodsSupplierFreightVo> supplierFreightList = ipGoodsSuppliersVo.getSupplierFreightList();
                if (supplierFreightList != null && supplierFreightList.size() > 0) {
                    supplierFreightListDto = new ArrayList<>();
                    IpGoodsSupplierFreightDto supplierFreightDto = null;
                    for (IpGoodsSupplierFreightVo ipGoodsSupplierFreightVo : supplierFreightList) {
                        supplierFreightDto = new IpGoodsSupplierFreightDto();
                        MapObjUtil.copy(ipGoodsSupplierFreightVo, supplierFreightDto);
                        supplierFreightListDto.add(supplierFreightDto);
                    }
                    ipGoodsSuppliersDto.setSupplierFreightList(supplierFreightListDto);

                }

                suppliersDtoList.add(ipGoodsSuppliersDto);
            }
            goodsAddDto.setSupplierList(suppliersDtoList);
        }

        this.saveGoods(goodsAddDto);
    }

    /**
     * 扣减库存
     *
     * @Param: * @param ipGoodsStockDto
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/11/2 15:47
     */
    @Override
    public void deductionGoodsStock(IpGoodsStockDto ipGoodsStockDto) throws Exception {

        String goodsType = ipGoodsStockDto.getGoodsType();

        //当为零售商品时，需要扣减库存
        if (StringUtils.isNotEmpty(goodsType) && "GR".equals(goodsType)) {
            String attrId = ipGoodsStockDto.getAttrId();
            //判断是否是规格型号下单
            if (StringUtils.isNotEmpty(attrId)) {
                ipGoodsPriceMapper.updateGoodsStock(ipGoodsStockDto);
            }
            goodsMapper.updateGoodsStock(ipGoodsStockDto);
        }
        if (StringUtils.isNotEmpty(ipGoodsStockDto.getType()) && "success".equals(ipGoodsStockDto.getType())) {

            String str = initsalenum + ipGoodsStockDto.getGoodsId() + "," + ipGoodsStockDto.getGoodsAmount();
            JSONObject jsonObject = new JSONObject();
            try {
                String s = HttpXmlClient.post1(jsonObject.toString(), str);
                logger.info(s);
            } catch (Exception e) {
                e.printStackTrace();
            }


        }


    }

    //商品新增与直接提交
    private void initGoodsSap(String type, String goodsId, GoodsAddDto goodsAddDto) throws Exception {
        List<IpGoodsManagesVo> goodsManagesVos = new ArrayList<>();

        //现在
        IpGoodsManageVo xGoodsManage = this.getXGoodsManage(goodsAddDto.getGoodsMap());
        xGoodsManage.setGoodsId(goodsId);
        IpGoodsManagesVo ipGoodsManagesVo = new IpGoodsManagesVo();
        ipGoodsManagesVo.setIntfType(type);
        ipGoodsManagesVo.setXgoodsManageVo(xGoodsManage);
        List<IpGoodsSapPriceVo> xgoodsPrice = this.getXgoodsPrice(goodsAddDto);
        ipGoodsManagesVo.setXgoodsPrice(xgoodsPrice);
        String categoryId = xGoodsManage.getCategoryId();
        List<IpGoodsManageSuppliersVo> goodsManageSupplier = ipGoodsSapService.getGoodsManageSupplier(categoryId);
        ipGoodsManagesVo.setSuppliersVos(goodsManageSupplier);
        goodsManagesVos.add(ipGoodsManagesVo);

        ipGoodsSapService.initGoodsDetailList(goodsManagesVos);
    }

    //
    @Override
    public void initGoodsSapId(String type, String goodsId, String goodsType) throws Exception {
        List<IpGoodsManagesVo> goodsManagesVos = new ArrayList<>();
        //现在
        IpGoodsManageVo goodsManage = ipGoodsSapService.getGoodsManage(goodsId);
        goodsManage.setGoodsId(goodsId);
        goodsManage.setValid(goodsType);
        IpGoodsManagesVo ipGoodsManagesVo = new IpGoodsManagesVo();
        ipGoodsManagesVo.setIntfType(type);
        ipGoodsManagesVo.setXgoodsManageVo(goodsManage);
        //
        if (goodsType != null && "X".equals(goodsType)) {

        } else {
            //
            List<IpGoodsSapPriceVo> xgoodsPrice = ipGoodsSapService.getGoodsPrice(goodsId);
            ipGoodsManagesVo.setXgoodsPrice(xgoodsPrice);
            String categoryId = goodsManage.getCategoryId();
            List<IpGoodsManageSuppliersVo> goodsManageSupplier = ipGoodsSapService.getGoodsManageSupplier(categoryId);
            if (goodsManageSupplier != null && goodsManageSupplier.size() > 0) {
                for (IpGoodsManageSuppliersVo manageSuppliersVo : goodsManageSupplier) {
                    manageSuppliersVo.setGoodsId(goodsId);
                }
            }

            ipGoodsManagesVo.setSuppliersVos(goodsManageSupplier);
        }
        goodsManagesVos.add(ipGoodsManagesVo);
        //同步sap
        ipGoodsSapService.initGoodsDetailList(goodsManagesVos);
    }

    private List<IpGoodsSapPriceVo> getXgoodsPrice(GoodsAddDto goodsAddDto) {
        List<IpGoodsSapPriceVo> sapPriceVos = new ArrayList<>();
        List<IpGoodsPriceDto> priceDtoList = goodsAddDto.getPriceDtoList();
        IpGoodsSapPriceVo goodsSapPriceVo = null;
        for (IpGoodsPriceDto ipGoodsPriceDto : priceDtoList) {
            goodsSapPriceVo = new IpGoodsSapPriceVo();
            String custType = ipGoodsPriceDto.getCustType();
            if (StringUtils.isNotEmpty(custType) && custType.equals("NB")) {
                goodsSapPriceVo.setCompanyId(ipGoodsPriceDto.getCompanyId());
                goodsSapPriceVo.setHsprice(ipGoodsPriceDto.getTaxPrice().toString());
                goodsSapPriceVo.setNoHsPrice(ipGoodsPriceDto.getNotaxPrice().toString());
                sapPriceVos.add(goodsSapPriceVo);
            }
        }
        return sapPriceVos;
    }

    private IpGoodsManageVo getXGoodsManage(Map<String, Object> goodsMap) {

        IpGoodsManageVo ipGoodsManageVo = new IpGoodsManageVo();
        ipGoodsManageVo.setTaxRate(this.getMapValue(goodsMap, "taxrate"));
        ipGoodsManageVo.setUnit(this.getMapValue(goodsMap, "unit"));
        ipGoodsManageVo.setPrice(this.getMapValue(goodsMap, "allPrice"));
        ipGoodsManageVo.setTaxFreePrice(this.getMapValue(goodsMap, "allOutPrice"));
        ipGoodsManageVo.setGoodsNum(this.getMapValue(goodsMap, "goodsnum"));
        ipGoodsManageVo.setGoodsName(this.getMapValue(goodsMap, "name"));
        ipGoodsManageVo.setSpecifications(this.getMapValue(goodsMap, "FTNPSpecificationsModel"));
        ipGoodsManageVo.setBillGoodsName(this.getMapValue(goodsMap, "billGoodsName"));
        ipGoodsManageVo.setBillTypeCode(this.getMapValue(goodsMap, "billTypeCode"));
        ipGoodsManageVo.setTaxTypeCode(this.getMapValue(goodsMap, "taxTypeCode"));
        ipGoodsManageVo.setContractId(this.getMapValue(goodsMap, "contractid"));
        ipGoodsManageVo.setSalesarea(this.getMapValue(goodsMap, "salesarea"));
        ipGoodsManageVo.setCategoryId(this.getMapValue(goodsMap, "FTNPThreeCategoryCode"));

        return ipGoodsManageVo;
    }

    private String getMapValue(Map<String, Object> goodsMap, String key) {
        String value = "";
        if (goodsMap.containsKey(key) && goodsMap.get(key) != null) {
            value = goodsMap.get(key).toString();
        }
        return value;
    }

    /**
     * 根据商品ID获取规格型号
     *
     * @param goodsSpecificationDto
     * @return
     * @throws Exception
     */
    @Override
    public GoodsSpecificationVo getGoodsSpecification(GoodsSpecificationDto goodsSpecificationDto) throws Exception {
        GoodsSpecificationVo goodsSpecificationVo = new GoodsSpecificationVo();
        /**
         *  1.首先根据客户类型获取所有的规格型号
         *  2.获取该客户类型的默认规格型号
         *  3.拼接规格型号的数据
         */
        String goodsId = goodsSpecificationDto.getGoodsId();
        long longGoodsId = Long.parseLong(goodsId);
        List<IpGoodsPrice> outGoodsPriceList = ipGoodsPriceMapper.getOutGoodsPriceList(longGoodsId);
        if (outGoodsPriceList != null && outGoodsPriceList.size() > 0) {
            List<IpGoodPriceVo> goodsPricevos = new ArrayList<IpGoodPriceVo>();
            IpGoodPriceVo goodPriceVo = null;
            Set<String> colorSet = new HashSet<>();
            Set<String> sizeSet = new HashSet<>();
            Set<String> otherSet = new HashSet<>();
            Set<String> modelSet = new HashSet<>();
            for (IpGoodsPrice ipGoodsPrice : outGoodsPriceList) {

                if (StringUtils.isNotEmpty(ipGoodsPrice.getSpeciColor())) {
                    colorSet.add(ipGoodsPrice.getSpeciColor());
                }
                if (StringUtils.isNotEmpty(ipGoodsPrice.getSpeciSize())) {
                    sizeSet.add(ipGoodsPrice.getSpeciSize());
                }
                if (StringUtils.isNotEmpty(ipGoodsPrice.getSpeciOther())) {
                    otherSet.add(ipGoodsPrice.getSpeciOther());
                }
                if (StringUtils.isNotEmpty(ipGoodsPrice.getSpeciModel())) {
                    modelSet.add(ipGoodsPrice.getSpeciModel());
                }


                goodPriceVo = new IpGoodPriceVo();
                this.commonBuild(goodPriceVo, ipGoodsPrice);
                goodPriceVo.setModelGoodsId(ipGoodsPrice.getId().toString());
                goodsPricevos.add(goodPriceVo);
            }
            goodsSpecificationVo.setPriceVos(goodsPricevos);
            List<GoodsSpecificationTreeVo> treeVos = new ArrayList<>();
            GoodsSpecificationTreeVo goodsSpecificationTreeVo = null;

            if (colorSet != null && colorSet.size() > 0) {

                List<GoodsSpecificationModelVo> specification = this.getSpecification(colorSet);
                if (specification != null && specification.size() > 0) {
                    goodsSpecificationTreeVo = new GoodsSpecificationTreeVo();
                    goodsSpecificationTreeVo.setK("颜色");
                    goodsSpecificationTreeVo.setV(specification);
                    goodsSpecificationTreeVo.setK_s("s1");
                    treeVos.add(goodsSpecificationTreeVo);
                }
            }
            if (modelSet != null && modelSet.size() > 0) {

                List<GoodsSpecificationModelVo> specification = this.getSpecification(modelSet);
                if (specification != null && specification.size() > 0) {
                    goodsSpecificationTreeVo = new GoodsSpecificationTreeVo();
                    goodsSpecificationTreeVo.setK("型号");
                    goodsSpecificationTreeVo.setV(specification);
                    goodsSpecificationTreeVo.setK_s("s2");
                    treeVos.add(goodsSpecificationTreeVo);
                }
            }
            if (sizeSet != null && sizeSet.size() > 0) {
                List<GoodsSpecificationModelVo> specification = this.getSpecification(sizeSet);
                if (specification != null && specification.size() > 0) {
                    goodsSpecificationTreeVo = new GoodsSpecificationTreeVo();
                    goodsSpecificationTreeVo.setK("尺码");
                    goodsSpecificationTreeVo.setV(specification);
                    goodsSpecificationTreeVo.setK_s("s3");
                    treeVos.add(goodsSpecificationTreeVo);
                }
            }
            if (otherSet != null && otherSet.size() > 0) {
                List<GoodsSpecificationModelVo> specification = this.getSpecification(otherSet);
                if (specification != null && specification.size() > 0) {

                    goodsSpecificationTreeVo = new GoodsSpecificationTreeVo();
                    goodsSpecificationTreeVo.setK("其他");
                    goodsSpecificationTreeVo.setV(specification);
                    goodsSpecificationTreeVo.setK_s("s4");
                    treeVos.add(goodsSpecificationTreeVo);
                }
            }
            goodsSpecificationVo.setTree(treeVos);
        }
        IpGoodsPrice outGoodsPriceNbyg = ipGoodsPriceMapper.getOutGoodsPriceNbyg(longGoodsId);
        IpGoodPriceVo goodPriceVo = null;
        if (outGoodsPriceNbyg != null) {
            goodPriceVo = new IpGoodPriceVo();
            this.commonBuild(goodPriceVo, outGoodsPriceNbyg);
            goodPriceVo.setModelGoodsId(outGoodsPriceNbyg.getId().toString());
            goodsSpecificationVo.setDefaultPrice(goodPriceVo);
        }
        return goodsSpecificationVo;

    }

    /**
     * Gets specification.
     *
     * @param stringSet the string set
     * @return the specification
     */
    public List<GoodsSpecificationModelVo> getSpecification(Set<String> stringSet) {
        List<GoodsSpecificationModelVo> goodsSpecificationModelVos = new ArrayList<GoodsSpecificationModelVo>();
        GoodsSpecificationModelVo specificationModelVo = null;
        for (String string : stringSet) {
            specificationModelVo = new GoodsSpecificationModelVo();
            specificationModelVo.setId(string);
            specificationModelVo.setName(string);
            goodsSpecificationModelVos.add(specificationModelVo);
        }
        return goodsSpecificationModelVos;
    }


    /**
     * Gets process business.
     *
     * @param goodsId the goods id
     * @param userid  the userid
     * @return the process business
     * @throws Exception the exception
     */
    public Map<String, Object> getProcessBusiness(Long goodsId, String userid) throws Exception {
        Map<String, Object> map = new HashMap<>();
        GoodsDeatilVo goodsDeatilVo = this.getGoodsDetailByGoodsId(goodsId, null);

        Map<String, Object> goodsMap = goodsDeatilVo.getGoodsMap();
        //商品类型
        String ftnpjc = "XM";
        String categoryId = null;
        String salesarea = null;
        if (goodsMap != null) {
            if (goodsMap.containsKey("FTNPJC")) {
                if (StringUtils.isNotEmpty(goodsMap.get("FTNPJC").toString())) {
                    ftnpjc = goodsMap.get("FTNPJC").toString();
                }
            }
            if (goodsMap.containsKey("FTNPThreeCategoryCode")) {
                if (StringUtils.isNotEmpty(goodsMap.get("FTNPThreeCategoryCode").toString())) {
                    categoryId = goodsMap.get("FTNPThreeCategoryCode").toString();
                }
            }
        }
        map.put("GOODSTYPE", ftnpjc);

        //获取当前供应商品类
        String salacode = this.getSalacode(userid, categoryId);
        String categoryTreeType = ipGoodsPriceMapper.getCategoryTreeType(salacode);
        // 获取供货类型
        map.put("SUPPORTTYPE", categoryTreeType);
        //判断是否包含外部的价格
        map.put("ISINCLUDEWB", this.isPriceIncludeNb(goodsDeatilVo));

        //获取销售的所有区域
        String priceSalar = this.getPriceSalar(goodsDeatilVo);

        Map<String, Object> areaPersion = this.getAreaPersion(null, priceSalar);
        map.putAll(areaPersion);

        return map;
    }

    /**
     * 区采商品
     *
     * @return
     */
    private Map<String, Object> getAreaPersion(String categoryTreeType, String priceSalar) {
        //获取区采的审批人
        Map<String, Object> map = new HashMap<>(16);
        //判断是否存在项目
        //获取销售的类型
        String saleType = ipGoodsPriceMapper.getCategoryTreeType(priceSalar);
        String priceSalarPersion = this.getParentPriceSalar(priceSalar);
        //判断价格区域是项目

        //获取项目
        Integer categoryXmCompany = ipGoodsPriceMapper.getCategoryXmCompany(priceSalar);
        String userPersionXm = ipGoodsPriceMapper.getUserPersionXm(priceSalar);
        map.put("PROJECTCODE", userPersionXm);
        map.put("PROJECTNUM", categoryXmCompany.toString());

        //获取城市
        Integer categoryCityCompany = ipGoodsPriceMapper.getCategoryCityCompany(priceSalarPersion);
        String userPersionCity = ipGoodsPriceMapper.getUserPersionCity(priceSalarPersion);
        map.put("CITYPERSIONCODE", userPersionCity);
        map.put("CITYNUM", categoryCityCompany.toString());

        //获取区域
        String userPersionArea = ipGoodsPriceMapper.getUserPersionArea(priceSalarPersion);
        map.put("AREAPERSION", userPersionArea);

        map.put("SALETYPE", saleType);

        return map;
    }


    //判断是否非内部
    private String isPriceIncludeNb(GoodsDeatilVo goodsDeatilVo) {
        //
        String flag = "N";
        List<IpGoodsPriceVo> priceList = goodsDeatilVo.getPriceList();
        if (priceList != null && priceList.size() > 0) {
            for (IpGoodsPriceVo ipGoodsPriceVo : priceList) {
                String custType = ipGoodsPriceVo.getCustType();
                if (!"NB".equals(custType)) {
                    flag = "Y";
                    break;
                }
            }
        }

        return flag;
    }

    private String getPriceSalar(GoodsDeatilVo goodsDeatilVo) {
        //
        List<IpGoodsPriceVo> priceList = goodsDeatilVo.getPriceList();
        Set<String> companySet = new HashSet<String>();
        if (priceList != null && priceList.size() > 0) {
            for (IpGoodsPriceVo ipGoodsPriceVo : priceList) {
                String custType = ipGoodsPriceVo.getCustType();
                if ("NB".equals(custType)) {
                    String companyId = ipGoodsPriceVo.getCompanyId();
                    companySet.addAll(this.getCompanyId(companyId));
                }
            }
        }

        return String.join(",", companySet);
    }

    private Set<String> getCompanyId(String companyId) {
        String[] split = companyId.split(",");
        Set<String> companySet = new HashSet<String>();
        if (split != null && split.length > 0) {
            for (String s : split) {
                companySet.add(s);
            }
        }
        return companySet;
    }

    private String getParentPriceSalar(String companyId) {
        String[] split = companyId.split(",");
        Set<String> companySet = new HashSet<String>();
        if (split != null && split.length > 0) {
            for (String s : split) {
                String companyParent = ipGoodsPriceMapper.getCompanyParent(s);
                String[] splits = companyParent.split(",");
                for (String split1 : splits) {
                    companySet.add(split1);
                }

            }
        }
        return String.join(",", companySet);
    }


    /**
     * 获取品类区域
     *
     * @param userid
     * @param categoryId
     * @return
     */
    private String getSalacode(String userid, String categoryId) {
        List<String> projectCodeMain = ipGoodsPriceMapper.getCategorySalare(categoryId, userid);
        String saleearea = String.join(",", projectCodeMain);

        return saleearea;
    }
    //判断供货的区域

    private List<String> getProjectCodeMain(String salesarea) {
        String[] split = salesarea.split(",");
        List<String> arrayList = null;
        if (split != null && split.length > 0) {
            HashSet<String> objects = new HashSet<String>();
            for (String s : split) {
                if (objects.contains(s)) {
                    continue;
                }
                objects.add(s);
            }
            arrayList = new ArrayList<String>(objects);
        }
        return arrayList;
    }
}
