package com.ekingwin.bas.cloud.cart.service.impl;

import com.ekingwin.bas.cloud.cart.dao.CartMapper;
import com.ekingwin.bas.cloud.cart.dao.entity.Cart;
import com.ekingwin.bas.cloud.cart.dto.CartDto;
import com.ekingwin.bas.cloud.cart.dto.CartGoodsDto;
import com.ekingwin.bas.cloud.cart.dto.IdleGoodsListDto;
import com.ekingwin.bas.cloud.cart.service.IpCartService;
import com.ekingwin.bas.cloud.dao.IpcCartMapper;
import com.ekingwin.bas.cloud.entity.IpcCart;
import com.ekingwin.bas.cloud.infra.dto.BaseResult;
import com.ekingwin.bas.cloud.infra.utils.common.ResultUtils;
import com.ekingwin.bas.cloud.util.Result;
import com.ekingwin.bas.cloud.infra.dto.BaseResult;
import com.ekingwin.bas.cloud.util.Result;
import com.ekingwin.bas.cloud.util.StringUtils;
import com.ekingwin.bas.cloud.vo.IpOrderGoodsDetailVo;
import com.github.pagehelper.PageHelper;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author kan
 */
@Service
@Primary
@Transactional(rollbackFor = Exception.class)
public class IpCartServiceImpl implements IpCartService {
    @Autowired
    private IpcCartMapper ipcCartMapper;
    @Autowired
    private CartMapper cartMapper;
    @Value("${goods.baseurl}")
    private String basPath;

    /**
     * 加入购物车
     * 1.判断商品是否失效
     *
     * @param cartDto
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public Result saveCart(CartDto cartDto) {
        //判断用户是否有权限操作
        if (0 != cartMapper.selectResult(cartDto.getUserid())) {
            return Result.error("您无权限购买此商品，请找管理员添加权限或者更换其他商品");
        }

        //判断商品的状态
        Integer status = ipcCartMapper.getGoodsStatusById(cartDto.getGoodsid());
        if (status == null || status != 1) {
            return Result.error("该商品已下架，不能加入购物车！");
        }
        String goodsId = cartDto.getGoodsid();

        List<Cart> carts = cartMapper.queryDate(cartDto);

        //判断购物车商品是否已存在
        if (carts != null && carts.size() > 0) {
            Cart cartO = carts.get(0);
            //购物车已有数量
            BigDecimal ocount = new BigDecimal(cartO.getCount());
            try {
                this.checkGoodsStock(goodsId, cartDto.getCount(), ocount, cartDto.getAttrid());
            } catch (Exception e) {
                e.printStackTrace();
                return Result.error("");
            }
            //购物车现有数量
            BigDecimal bcount = new BigDecimal(cartDto.getCount());
            //计算总的数量
            BigDecimal total = ocount.add(bcount);
            cartO.setCount(total.toString());

            try {
                cartMapper.updateCart(cartO);
                return Result.ok("更新到购物车成功");
            } catch (Exception e) {
                e.printStackTrace();
                return Result.error("更新到购物车失败");
            }
        } else {

            try {
                cartMapper.insertCart(cartDto);
                return Result.ok("添加到购物车成功");
            } catch (Exception e) {
                e.printStackTrace();
                return Result.error("添加到购物车失败");
            }
        }
    }

    /**
     * 校验库存
     *
     * @param count
     * @Param: * @param goodsId
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/28 10:40
     */
    @Override
    public Result checkGoodsStock(String goodsId, String count, BigDecimal ocount, String attrid) {
        //本次数量
        BigDecimal bDecimal = new BigDecimal(count);
        bDecimal = bDecimal.add(ocount);
        IpOrderGoodsDetailVo goodsStock = null;
        if (StringUtils.isNotEmpty(attrid)) {
            goodsStock = ipcCartMapper.getGoodsStockAttrid(goodsId, attrid);
        } else {
            //获取库存
            goodsStock = ipcCartMapper.getGoodsStock(goodsId);
        }


        if (!"1".equals(goodsStock.getState())) {
            return Result.error(goodsStock.getGoodsname() + "商品已下架");
        }
        BigDecimal stock = new BigDecimal(goodsStock.getStock());
        if (stock.compareTo(bDecimal) == -1) {
            return Result.error(goodsStock.getGoodsname() + "商品库存不足");
        }
        return null;
    }

    /**
     * 购物车列表
     *
     * @param cartDto
     * @return
     * @throws Exception
     */
    @Override
    public List<Cart> cartList(CartDto cartDto) throws Exception {
        List<Cart> list = cartMapper.queryDate(cartDto);
        if (list != null && list.size() > 0) {
            for (Cart c : list) {
                CartGoodsDto cartGoodsDtos = cartMapper.selectGoodsDetailAndMessage(c.getGoodsid());
                if (cartGoodsDtos != null) {
                    cartGoodsDtos.setPath(basPath +cartGoodsDtos.getPath());
                    cartGoodsDtos.setPrice(c.getPrice()+"");
                    c.setCartGoodsDtos(cartGoodsDtos);
                    c.setSupplierid(cartGoodsDtos.getSupplierid());
                    c.setSuppliername(cartGoodsDtos.getSuppliername());
                }
            }

        }
        return list;
    }

    @Override
    public Map<String, List<Cart>> cartList1(CartDto cartDto) throws Exception {
        List<Cart> list = cartMapper.queryDate(cartDto);
        Map<String, List<Cart>> resultMap = new HashMap<>();
        if (list != null && list.size() > 0) {
            for (Cart c : list) {
                Map<Object, Object> map = cartMapper.selectGoodsDetail(c.getGoodsid());
                if (map != null && !map.isEmpty()) {
                    map.put("path", basPath + map.get("path"));
                    map.put("price", c.getPrice());
                    map.put("count", c.getCount());
                    c.setGoodsMap(map);
                    c.setSupplierid(map.get("supplierid").toString());
                    c.setSuppliername(map.get("suppliername").toString());
                }
            }

            //同一供应商的商品，放入一个模块
            try {
                for (Cart cart : list) {
                    if (resultMap.containsKey(cart.getSupplierid())) {//map中异常批次已存在，将该数据存放到同一个key（key存放的是异常批次）的map中
                        resultMap.get(cart.getSupplierid()).add(cart);
                    } else {//map中不存在，新建key，用来存放数据
                        List<Cart> tmpList = new ArrayList<Cart>();
                        tmpList.add(cart);
                        resultMap.put(cart.getSupplierid(), tmpList);
                    }
                }
            } catch (Exception e) {
                throw new Exception("按照异常批次号对已开单数据进行分组时出现异常", e);
            }
        }
        return resultMap;
    }

    /**
     * 购物车个数
     *
     * @param userid
     * @return
     * @throws Exception
     */
    @Override
    public Integer cartCount(String userid) throws Exception {
        CartDto cartDto = new CartDto();
        cartDto.setUserid(userid);
        List<Cart> list = cartMapper.queryDate(cartDto);
        Integer count = 0;
        if (list != null && list.size() > 0) {
            count = list.size();
        }
        return count;
    }

    /**
     * 提交订单成功，清理购物车中信息
     *
     * @param ids
     * @return
     */
    @Override
    public BaseResult deletecart(String ids) {
        List<IpcCart> carts = ipcCartMapper.getGoodsid(ids);
        IpOrderGoodsDetailVo goodsStock = null;
        for (int i = 0; i < carts.size(); i++) {
            //购物车数量
            BigDecimal ocount = new BigDecimal(carts.get(i).getCount());
            //获取库存
            goodsStock = ipcCartMapper.getGoodsStock(carts.get(i).getGoodsid());
            //商品库现有库存量
            BigDecimal stock = new BigDecimal(goodsStock.getStock());
            //释放购物车库存
            BigDecimal total = stock.subtract(ocount);
            try {
                ipcCartMapper.updateGoodsStock(carts.get(i).getGoodsid(), total.toString());
            } catch (Exception e) {
                e.printStackTrace();
                return ResultUtils.build("释放购物车内存失败！");
            }
        }
        //删除（批量删除）购物车
        try {
            ipcCartMapper.deleteByCarIds(ids);
        } catch (Exception e) {
            e.printStackTrace();
            return ResultUtils.build("已释放购物车内存，但购物车删除失败！");
        }
        return ResultUtils.build("删除成功！");
    }

    /**
     * 批量删除购物车
     *
     * @param ids
     * @return
     */
    @Override
    public BaseResult batchRemoveCart(String ids) {
        //删除（批量删除）购物车
        try {
            ipcCartMapper.deleteByCarIds(ids);
        } catch (Exception e) {
            e.printStackTrace();
            return ResultUtils.build("购物车删除失败！");
        }
        return ResultUtils.build("删除成功！");
    }

    /**
     * 购物车页面结算
     *
     * @param goodsListDto
     * @return
     */
    @Override
    public Result cartAdjustment(List<IdleGoodsListDto> goodsListDto) {
        for (int z = 0; z < goodsListDto.size(); z++) {
            //查询当前的goodsid对应的物料编码
            String mdmcode = ipcCartMapper.querymdmcode(goodsListDto.get(z).getGoodsid());
            //查询当前商品库存量
            String stock = ipcCartMapper.querystock(goodsListDto.get(z).getGoodsid());
            BigInteger stock1 = new BigInteger(stock);
            //查询出物料编码对应的闲置商品
            List<IdleGoodsListDto> listDtos = ipcCartMapper.queryidlegoods(mdmcode);
            //查询出物料编码对应的非闲置商品
            List<IdleGoodsListDto> listDto = ipcCartMapper.queryidlegood(mdmcode);
            //查询当前是闲置商品还是非闲置商品
            String idlegoods = ipcCartMapper.queryidlegoodss(goodsListDto.get(z).getGoodsid());
            if (idlegoods.equals("否") && !listDtos.isEmpty()) {
                for (int i = 0; i < listDtos.size(); i++) {
                    BigInteger momout = new BigInteger("1");
                    if (listDtos.get(i).getStock().compareTo(momout) == -1) {
                        if (listDtos.size() - 1 == i) {
                            BigInteger momout2 = new BigInteger("0");
                            BigInteger count = goodsListDto.get(z).getCount();
                            BigInteger stock2 = stock1.subtract(count);
                            //比较是否小于库存值
                            if (stock2.compareTo(momout2) == -1) {
                                for (int j = 0; j < listDto.size(); j++) {
                                    BigInteger momout3 = new BigInteger("0");
                                    BigInteger stock3 = listDto.get(j).getStock();
                                    BigInteger count2 = goodsListDto.get(z).getCount();
                                    BigInteger stock4 = stock3.subtract(count2);
                                    //比较是否小于库存值
                                    if (stock4.compareTo(momout3) == -1) {
                                        if (listDto.size() - 1 == j) {
                                            return Result.error("该商品物种库存全部不足，请换商品购买");
                                        }
                                    } else {
                                        return Result.error("商品编号为"+goodsListDto.get(z).getGoodsid() +"的供应商提供的商品库存不足，请换家购买此商品");
                                    }
                                }
                            }
                        }
                    } else {
                        return Result.error("库存中存在闲置商品，请先购买闲置商品");
                    }
                }
            } else if (idlegoods.equals("否") && listDtos.isEmpty()) {
                BigInteger momout2 = new BigInteger("0");
                BigInteger count = goodsListDto.get(z).getCount();
                BigInteger stock2 = stock1.subtract(count);
                //比较是否小于库存值
                if (stock2.compareTo(momout2) == -1) {
                    for (int j = 0; j < listDto.size(); j++) {
                        BigInteger momout3 = new BigInteger("0");
                        BigInteger stock3 = listDto.get(j).getStock();
                        BigInteger count2 = goodsListDto.get(z).getCount();
                        BigInteger stock4 = stock3.subtract(count2);
                        //比较是否小于库存值
                        if (stock4.compareTo(momout3) == -1) {
                            if (listDto.size() - 1 == j) {
                                return Result.error("该商品物种库存全部不足，请换商品购买");
                            }
                        } else {
                            return Result.error("您购买的商品中有供应商提供的商品库存不足，请换家购买此商品");
                        }
                    }
                }
            } else if (idlegoods.equals("是") && listDto.isEmpty()) {
                BigInteger momout2 = new BigInteger("0");
                BigInteger count = goodsListDto.get(z).getCount();
                BigInteger stock2 = stock1.subtract(count);
                //比较是否小于库存值
                if (stock2.compareTo(momout2) == -1) {
                    for (int j = 0; j < listDtos.size(); j++) {
                        BigInteger momout3 = new BigInteger("1");
                        BigInteger stock3 = listDtos.get(j).getStock();
                        //比较是否小于库存值
                        if (stock3.compareTo(momout3) == -1) {
                            if (listDtos.size() - 1 == j) {
                                return Result.error("该商品物种库存全部不足，请换商品购买");
                            }
                        } else {
                            return Result.error("您购买的商品中有供应商提供的商品库存不足，请换家购买此商品");
                        }
                    }
                }
            } else if (idlegoods.equals("是") && !listDto.isEmpty()) {
                BigInteger momout2 = new BigInteger("0");
                BigInteger count = goodsListDto.get(z).getCount();
                BigInteger stock2 = stock1.subtract(count);
                //比较是否小于库存值
                if (stock2.compareTo(momout2) == -1) {
                    for (int j = 0; j < listDtos.size(); j++) {
                        BigInteger momout3 = new BigInteger("1");
                        BigInteger stock3 = listDtos.get(j).getStock();
                        //比较是否小于库存值
                        if (stock3.compareTo(momout3) == -1) {
                            if (listDtos.size() - 1 == j) {
                                for (int i = 0; i < listDto.size(); i++) {
                                    BigInteger momout4 = new BigInteger("1");
                                    BigInteger stock4 = listDto.get(i).getStock();
                                    if (stock4.compareTo(momout4) == -1) {
                                        if (listDto.size() - 1 == i) {
                                            return Result.error("该商品物种库存全部不足，请换商品购买");
                                        }
                                    } else {
                                        return Result.error("该闲置商品库存不足,请购买该商品的其他非闲置商品");
                                    }
                                }
                            }
                        } else {
                            return Result.error("您购买的商品中有供应商提供的商品库存不足，请换家购买此商品");
                        }
                    }
                }
            }
        }
        return Result.ok("下单成功");
    }

    /**
     * 购物车页面数量的增减
     *
     * @param goodsListDto
     * @return
     */
    @Override
    public Result cartAddCount(IdleGoodsListDto goodsListDto) {
        try {
            ipcCartMapper.UpdatecartAddCount(goodsListDto);
            return Result.ok();
        } catch (Exception e) {
            e.printStackTrace();
            return Result.error("购物车页面数量的增减失败");
        }
    }
}

