package com.ekingwin.bas.cloud.service.impl;

import com.ekingwin.bas.cloud.dao.IpOrderRejectedEntityMapper;
import com.ekingwin.bas.cloud.dao.IpOrderRejectedGoodsEntityMapper;
import com.ekingwin.bas.cloud.dto.IpOrderRejectGoodsConfirmDto;
import com.ekingwin.bas.cloud.dto.IpOrderRejectSearchDto;
import com.ekingwin.bas.cloud.dto.IpOrderRejectedGoodsDto;
import com.ekingwin.bas.cloud.dto.SendLogicMessageDto;
import com.ekingwin.bas.cloud.entity.IpOrderRejectedEntity;
import com.ekingwin.bas.cloud.entity.IpOrderRejectedGoodsEntity;
import com.ekingwin.bas.cloud.entity.IpcOrderAllgoods;
import com.ekingwin.bas.cloud.handler.BusinessException;
import com.ekingwin.bas.cloud.monitor.dto.IpOrderReconciliationDto;
import com.ekingwin.bas.cloud.rabbit.sender.LogicSender;
import com.ekingwin.bas.cloud.rabbit.sender.OrderRejectSender;
import com.ekingwin.bas.cloud.rabbit.sender.ReconciliationOrderSender;
import com.ekingwin.bas.cloud.reqno.utils.GenerateRequestCode;
import com.ekingwin.bas.cloud.service.IpOrderDeliverService;
import com.ekingwin.bas.cloud.service.IpOrderRejectedService;
import com.ekingwin.bas.cloud.service.IpcOrderService;
import com.ekingwin.bas.cloud.util.DateUtils;
import com.ekingwin.bas.cloud.util.MapObjUtil;
import com.ekingwin.bas.cloud.util.StringUtils;
import com.ekingwin.bas.cloud.vo.IpOrderRejectedGoodsVo;
import com.ekingwin.bas.cloud.vo.IpOrderRejectedVo;
import com.ekingwin.bas.cloud.vo.IpOrderVo;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @ClassName :
 * @description:
 * @author: zhangjie
 * @create: 2019-10-16 14:40
 **/
@Service
@Primary
public class IpOrderRejectedServiceImpl implements IpOrderRejectedService {


    @Autowired
    private IpOrderRejectedEntityMapper rejectedEntityMapper;

    @Autowired
    private IpOrderRejectedGoodsEntityMapper rejectedGoodsEntityMapper;

    @Autowired
    private IpOrderDeliverService ipOrderDeliverService;

    @Autowired
    private IpcOrderService ipcOrderService;

    @Autowired
    private ReconciliationOrderSender reconciliationOrderSender;


    @Value("${goods.baseurl}")
    private String basPath;

    @Autowired
    private LogicSender logicSender;

    @Autowired
    private OrderRejectSender orderRejectSender;

    /**
    * 保存退货信息
     * 1.保存退货单信息
     * 2.更改商品的商品的状态
    * @Param:  * @param ipOrderRejectedGoodsDto
    * @return: java.lang.Long
    * @Author: zhangjie
    * @Date: 2019/10/16 16:00
    */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public Long saveReject(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto) throws Exception{

        //验证退货状态
        this.checkOrderReject(ipOrderRejectedGoodsDto);
        //保存退货单数据
        Long rejectId=this.saveRejectMain(ipOrderRejectedGoodsDto);
        //保存退货单商品数据
        this.saveRejectGoods(ipOrderRejectedGoodsDto,rejectId);
        //同步数据到运营表
        this.initReceiverMonitor(ipOrderRejectedGoodsDto);
        return rejectId;
    }

    /**
    * 校验退货
    * @Param:  * @param ipOrderRejectedGoodsDto
    * @return: void
    * @Author: zhangjie
    * @Date: 2019/11/6 18:31
    */
    private void checkOrderReject(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto){
        String orderGoodsId = ipOrderRejectedGoodsDto.getOrderGoodsId().toString();
        //根据订单商品id 获取详情
        IpcOrderAllgoods orderGoods = ipcOrderService.getOrderGoods(orderGoodsId);
        if (orderGoods !=null){
            String state = orderGoods.getState();
            //判断订单状态是否是退货中、已退货、已收货
            if (StringUtils.isNotEmpty(state) && ("退货中".equals(state) || "已退货".equals(state)  || "已收货".equals(state) )){
                throw  new BusinessException(500, "此商品已在退货确认中或已退货完成");
            }
        }
    }

    /** 
    * 保存主表
    */
    private Long saveRejectMain(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto) throws Exception{
        IpOrderRejectedEntity rejectedEntity = new IpOrderRejectedEntity();
        //获取退货单号
        //String rejectCode = rejectedEntityMapper.getRejectCode();
        String rejectCode= GenerateRequestCode.requestCode("orderReject",
                GenerateRequestCode.YEARMONTHDAY,4,"THDD");
        rejectedEntity.setCreateId(ipOrderRejectedGoodsDto.getUserid());
        rejectedEntity.setCreateTime(new Date());
        rejectedEntity.setCreateUser(ipOrderRejectedGoodsDto.getUserName());
        rejectedEntity.setRejectedCode(rejectCode);
        rejectedEntityMapper.insertSelective(rejectedEntity);
        String courierLogo = ipOrderRejectedGoodsDto.getCourierLogo();
        //判断是否存在物流单号和能否匹配到物流公司，若可以则发送快递100进行订阅
        if (StringUtils.isNotEmpty(ipOrderRejectedGoodsDto.getLogisticsNumber()) && StringUtils.isNotEmpty(courierLogo)){
            this.sendLogicMessage(ipOrderRejectedGoodsDto,rejectCode);
        }
        return  rejectedEntity.getId();
    }

    /**
    * 保存退货商品表
    * @Param:  * @param ipOrderRejectedGoodsDto
     * @param rejectedId
    * @return: void
    * @Author: zhangjie
    * @Date: 2019/10/16 15:49
    */
    private void saveRejectGoods(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto,
                                 Long rejectedId) throws Exception{
        IpOrderRejectedGoodsEntity ipOrderRejectedGoodsEntity = new IpOrderRejectedGoodsEntity();
        MapObjUtil.copy(ipOrderRejectedGoodsDto,ipOrderRejectedGoodsEntity);
        ipOrderRejectedGoodsEntity.setRejectedId(rejectedId);
        ipOrderRejectedGoodsEntity.setStatus("退货中");
        rejectedGoodsEntityMapper.insertSelective(ipOrderRejectedGoodsEntity);

        //修改商品的状态为退货中
        this.updateOrderGoodsStatus(ipOrderRejectedGoodsDto.getOrderGoodsId(),"退货中");
    }

    /**
    * 修改商品的退货状态
    * @Param:  * @param id
     * @param status
    * @return: void
    * @Author: zhangjie
    * @Date: 2019/10/26 11:06
    */
    private void updateOrderGoodsStatus(Long id,String status){
        rejectedGoodsEntityMapper.updateOrderGoodsStatus(id, status);
    }

    /**
    * 退货确认
    * @Param:  * @param goodsConfirmDto
    * @return: void
    * @Author: zhangjie
    * @Date: 2019/10/26 17:04
    */
    @Override
    public void updateRejectGoodsConfim(IpOrderRejectGoodsConfirmDto goodsConfirmDto) throws Exception {
        //修改退货单状态为已退货
        this.updateOrderGoodsStatus(goodsConfirmDto.getOrderGoodsId(), "已退货");
        //修改退货单的状态
        rejectedGoodsEntityMapper.updateGoodsStatusByRejectId(goodsConfirmDto.getRejectedId(), "已退货");
        //获取退货单信息
        IpOrderRejectedVo orderRejectedDetail = this.getOrderRejectedDetail(goodsConfirmDto.getRejectedId().toString());

        //调用银联支付
        goodsConfirmDto.setRefundNo(orderRejectedDetail.getRejectCode());
        String jsonStr = JSONObject.fromObject(goodsConfirmDto).toString();
        orderRejectSender.send("rejected.payment.info", jsonStr);
    }

    /**
    * 查询退货单列表
    * @Param:  * @param ipOrderRejectSearchDto
    * @return: com.github.pagehelper.PageInfo<org.apache.poi.ss.formula.functions.T>
    * @Author: zhangjie
    * @Date: 2019/10/16 17:57
    */
    @Override
    public PageInfo<IpOrderRejectedVo> getOrderRejectedList(IpOrderRejectSearchDto ipOrderRejectSearchDto)
            throws Exception{
        PageHelper.startPage(ipOrderRejectSearchDto.getPagenum(),ipOrderRejectSearchDto.getPagesize());
        List<IpOrderRejectedVo> rejectedVoList=null;
        String isSupplier = ipOrderRejectSearchDto.getIsSupplier();
        //判断是否是供应商
        if (StringUtils.isNotEmpty(isSupplier) && "Y".equals(isSupplier)){
            //供应商获取退货单列表
            rejectedVoList=rejectedEntityMapper.getOrderRejectedListBySupplierId(ipOrderRejectSearchDto);
        } else{
            rejectedVoList=   rejectedEntityMapper.getOrderRejectedListByUserId(ipOrderRejectSearchDto);
        }
        //获取退货单商品数据
        if (rejectedVoList !=null && rejectedVoList.size()>0){
            for (IpOrderRejectedVo ipOrderRejectedVo : rejectedVoList) {
                String rejectedId=ipOrderRejectedVo.getRejectedId();
                ipOrderRejectSearchDto.setRejectedId(rejectedId);
                ipOrderRejectedVo.setGoodsList(this.getOrderRejectedGoods(ipOrderRejectSearchDto));
            }
        }
        PageInfo<IpOrderRejectedVo> pageInfo=new PageInfo<>(rejectedVoList);
        return  pageInfo;
    }

    /**
     * 获取送货单商品信息
     *
     * @return
     */
    private List<IpOrderRejectedGoodsVo> getOrderRejectedGoods(IpOrderRejectSearchDto ipOrderRejectSearchDto) {
        List<IpOrderRejectedGoodsVo> deliverGoodVoList =
                rejectedGoodsEntityMapper.getOrderRejectGood(ipOrderRejectSearchDto);
        for (IpOrderRejectedGoodsVo ipOrderReceiverGoodVo : deliverGoodVoList) {
            if (StringUtils.isNotEmpty(ipOrderReceiverGoodVo.getUrlPath())){
                ipOrderReceiverGoodVo.setUrlPath(basPath + ipOrderReceiverGoodVo.getUrlPath());
            }

        }

        return deliverGoodVoList;
    }

    /**
    * 获取退货单详情
    * @Param:  * @param rejectedId
    * @return: com.ekingwin.bas.cloud.vo.IpOrderRejectedVo
    * @Author: zhangjie
    * @Date: 2019/10/28 18:03
    */
    @Override
    public IpOrderRejectedVo getOrderRejectedDetail(String rejectedId)
            throws Exception {
        IpOrderRejectedVo orderRejectedDetail = rejectedEntityMapper.getOrderRejectedDetail(rejectedId);
        if (orderRejectedDetail != null) {
            String courierLogo = orderRejectedDetail.getCourierLogo();
            //判断存在物流公司编码，从快递100推送的物流信息查询物流信息
            if (courierLogo != null) {
                String logisticsNumber = orderRejectedDetail.getLogisticsNumber();
                orderRejectedDetail.setLogistics(ipOrderDeliverService.getLogistics(logisticsNumber));
            }

        }
        return orderRejectedDetail;
    }


    /**
     * 推送消息到快递100
     *
     * @Param: * @param ipOrderDeliver
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/24 20:07
     */
    private void sendLogicMessage(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto,String rejectedNum) throws Exception {
        SendLogicMessageDto sendLogicMessageDto = new SendLogicMessageDto();
        sendLogicMessageDto.setCompanycode(ipOrderRejectedGoodsDto.getCourierLogo());
        sendLogicMessageDto.setCompanyname(ipOrderRejectedGoodsDto.getCourierName());
        sendLogicMessageDto.setCouriernumber(ipOrderRejectedGoodsDto.getLogisticsNumber());
        sendLogicMessageDto.setState("退货");
        sendLogicMessageDto.setSendnumber(rejectedNum);
        //
        String jsonStr = JSONObject.fromObject(sendLogicMessageDto).toString();
        logicSender.send("rejected.logic.info", jsonStr);
    }


    /**
     * 同步退货信息到运营台账表里
     * @param ipOrderRejectedGoodsDto
     * @throws Exception
     */
    private void initReceiverMonitor(IpOrderRejectedGoodsDto ipOrderRejectedGoodsDto) throws Exception{
        List<IpOrderReconciliationDto> reconciliationDtos = new ArrayList<>();
        IpOrderReconciliationDto ipOrderReconciliationDto = new IpOrderReconciliationDto();

        Long orderGoodsId = ipOrderRejectedGoodsDto.getOrderGoodsId();
        ipOrderReconciliationDto.setOrderGoodsId(String.valueOf(orderGoodsId));
        ipOrderReconciliationDto.setReceiverNum("");
        ipOrderReconciliationDto.setOrderAccount("");
        BigDecimal receverNum= StringUtils.stringToBigDecimal(ipOrderRejectedGoodsDto.getAmount());
        if (receverNum==null){
            receverNum=new BigDecimal("0.00");
        }
        //退货数量
        ipOrderReconciliationDto.setAccount(ipOrderRejectedGoodsDto.getAmount());

        IpcOrderAllgoods orderGoods = ipcOrderService.getOrderGoods(String.valueOf(orderGoodsId));
        Long orderid = orderGoods.getOrderid();
        IpOrderVo orderDetaiil = ipcOrderService.getOrderDetaiil(orderid);
        ipOrderReconciliationDto.setGoodsId(orderGoods.getGoodsid().toString());
        ipOrderReconciliationDto.setGoodsName(orderGoods.getGoodsname());
        ipOrderReconciliationDto.setOrderId(orderid.toString());
        ipOrderReconciliationDto.setGoodsnum(orderGoods.getGoodsnum());
        ipOrderReconciliationDto.setOrderNum(orderDetaiil.getOrdernum());
        ipOrderReconciliationDto.setOrderPersion(orderDetaiil.getApplyUser());
        ipOrderReconciliationDto.setOrderTime(DateUtils.stringToDate(orderDetaiil.getApplyTime(),DateUtils.DATE_TIME_PATTERN));
        ipOrderReconciliationDto.setOrderPersionCode(orderDetaiil.getApplyCode());
        ipOrderReconciliationDto.setOrderPersion(orderDetaiil.getApplyUser());
        String price = orderGoods.getPrice();
        BigDecimal bigPrice=new BigDecimal(price);
        if (bigPrice==null){
            bigPrice= new BigDecimal("0.00");
        }
        ipOrderReconciliationDto.setOrderPrice(bigPrice);
        BigDecimal totalPrice = bigPrice.multiply(receverNum);
        ipOrderReconciliationDto.setTotalPrice(totalPrice);
        ipOrderReconciliationDto.setRate(orderGoods.getTaxrate());
        String rmbhsprice = orderGoods.getRmbhsprice();
        if (StringUtils.isEmpty(rmbhsprice)){
            rmbhsprice="0.00";
        }
        BigDecimal rmbhsPrice=new BigDecimal(rmbhsprice);
        if (rmbhsPrice==null){
            rmbhsPrice=new BigDecimal("0.00");
        }
        //计算供应商价格
        BigDecimal supplicePrice = rmbhsPrice.multiply(receverNum);
        ipOrderReconciliationDto.setPrice(bigPrice);
        ipOrderReconciliationDto.setSupplierPrice(rmbhsPrice);
        ipOrderReconciliationDto.setSupplierTotalPrice(supplicePrice);

        String rmbbhsprice = orderGoods.getRmbbhsprice();
        if (StringUtils.isEmpty(rmbbhsprice)){
            rmbbhsprice="0.00";
        }
        //原件
        BigDecimal rmbbhsPrice=new BigDecimal(rmbbhsprice);
        if (rmbbhsPrice==null){
            rmbbhsPrice=new BigDecimal("0.00");
        }
        String bhspriceStr = orderGoods.getBhsprice();
        if (StringUtils.isEmpty(bhspriceStr)){
            bhspriceStr="0.00";
        }
        BigDecimal bhsprice=new BigDecimal(bhspriceStr);
        if (bhsprice==null){
            bhsprice=new BigDecimal("0.00");
        }
        ipOrderReconciliationDto.setNotaxPrice(bhsprice);
        ipOrderReconciliationDto.setSupplierNotaxprice(rmbbhsPrice);
        BigDecimal total=new BigDecimal(orderGoods.getGoodstotalhsprice());
        ipOrderReconciliationDto.setOrderTotalPrice(total);
        //区分是退货还是收货
        ipOrderReconciliationDto.setType("TH");
        ipOrderReconciliationDto.setSupplierId(orderGoods.getSupplierid());
        ipOrderReconciliationDto.setSupplierName(orderGoods.getSupplier());
        ipOrderReconciliationDto.setSupplierNum(orderGoods.getSupplierid());
        ipOrderReconciliationDto.setOrderType(orderDetaiil.getOrderType());
        ipOrderReconciliationDto.setProfitMargin(totalPrice.subtract(supplicePrice));
        reconciliationDtos.add(ipOrderReconciliationDto);

        if (reconciliationDtos != null && reconciliationDtos.size() > 0) {
            String toString = JSONArray.fromObject(reconciliationDtos).toString();
            reconciliationOrderSender.send("order.monitororder.info", toString);
        }
    }



}
