package com.ekingwin.bas.cloud.service.impl;

import com.ekingwin.bas.cloud.config.OrderConfig;
import com.ekingwin.bas.cloud.dao.IpAdministrativePersonnelMapper;
import com.ekingwin.bas.cloud.dao.IpOrderInvoiceMapper;
import com.ekingwin.bas.cloud.dao.IpOrderMapper;
import com.ekingwin.bas.cloud.dto.*;
import com.ekingwin.bas.cloud.handler.BusinessException;
import com.ekingwin.bas.cloud.infra.service.BaseService;
import com.ekingwin.bas.cloud.process.dto.CreateProcessDto;
import com.ekingwin.bas.cloud.process.service.IpcProcessService;
import com.ekingwin.bas.cloud.service.*;
import com.ekingwin.bas.cloud.util.DateUtils;
import com.ekingwin.bas.cloud.util.StringUtils;
import com.ekingwin.bas.cloud.vo.*;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;

/**
 * @program: rcworkspace
 * @description:
 * @author: zhangjie
 * @create: 2019-06-30 18:45
 **/
@Service
@Primary
@Transactional(rollbackFor = Exception.class)
public class IpOrderServiceImpl extends BaseService implements IpOrderService {

    Logger logger = LoggerFactory.getLogger(IpOrderServiceImpl.class);

    @Autowired
    private IpOrderMapper ipOrderMapper;

    @Autowired
    private IpOrderInvoiceMapper ipOrderInvoiceMapper;


    @Autowired
    private IpOrderDrafterService ipOrderDrafterService;

    @Autowired
    private IpcOrderService ipcOrderService;

    @Autowired
    private IpcCartService ipcCartService;

    @Autowired
    private IpcProcessService ipcProcessService;

    @Value("${goods.baseurl}")
    private String basPath;


    @Autowired
    private IpAdministrativePersonnelMapper ipAdministrativePersonnelMapper;


    @Autowired
    private IpGoodsContractService ipGoodsContractService;

    @Autowired
    private OrderConfig orderConfig;

    @Autowired
    private IpCustAddressService ipCustAddressService;


    /**
     * 查询购物下单时，商品详情
     *
     * @return
     */
    @Override
    public IpOrderCarDetailVo getOrderCarDetail(IpOrderDetailDto ipOrderDetailDto) throws Exception {
        IpOrderCarDetailVo ipOrderCarDetailVo = new IpOrderCarDetailVo();
        //去采购获取详情
        if (StringUtils.isNotEmpty(ipOrderDetailDto.getCarId())) {
            ipOrderCarDetailVo = this.getOrderGoodCarDetail(ipOrderDetailDto);
        } else if (StringUtils.isNotEmpty(ipOrderDetailDto.getDraftId())) {
            //从草稿箱进入获取订单详情
            ipOrderCarDetailVo = ipOrderDrafterService.getOrderCarDetail(ipOrderDetailDto);
        } else if (StringUtils.isNotEmpty(ipOrderDetailDto.getOrderId()) ||
                StringUtils.isNotEmpty(ipOrderDetailDto.getInstid())) {
            ipOrderDetailDto.setUserid(null);
            ipOrderCarDetailVo = ipcOrderService.getOrderCarDetail(ipOrderDetailDto);
        }
        return ipOrderCarDetailVo;
    }

    /**
     * 去下单获取信息
     *
     * @param ipOrderDetailDto
     * @return
     */
    private IpOrderCarDetailVo getOrderGoodCarDetail(IpOrderDetailDto ipOrderDetailDto) throws Exception {
        IpOrderCarDetailVo ipOrderCarDetailVo = new IpOrderCarDetailVo();
        String userId = ipOrderDetailDto.getUserid();

        //获取默认地址
        IpCustOrderAddressVo custAddress = ipCustAddressService.getCustAddress(userId);

        if (custAddress != null) {
            IpOrderVo order = new IpOrderVo();
            this.commonBuild(order, custAddress);
            String email = custAddress.getEmail();
            ipOrderCarDetailVo.setEmail(email);
            ipOrderCarDetailVo.setOrder(order);
        }

        //获取商品信息
        List<IpOrderGoodsDetailVo> ipOrderGoodsDetailVos = ipOrderMapper.getOrderGoodsDetail(ipOrderDetailDto);
        if (ipOrderGoodsDetailVos != null && ipOrderGoodsDetailVos.size() > 0) {
            for (IpOrderGoodsDetailVo ipOrderGoodsDetailVo : ipOrderGoodsDetailVos) {
                if (StringUtils.isNotEmpty(ipOrderGoodsDetailVo.getUrlPath())) {
                    ipOrderGoodsDetailVo.setUrlPath(basPath + ipOrderGoodsDetailVo.getUrlPath());

                }
                String category = ipOrderGoodsDetailVo.getFtnpconfiguration();
                if (StringUtils.isEmpty(category)) {
                    category = "CG";
                }
                ipOrderGoodsDetailVo.setCategory(category);

            }
        }
        ipOrderCarDetailVo.setGoods(ipOrderGoodsDetailVos);
        //ipOrderCarDetailVo.set

        //获取开票信息
        String customerType = ipOrderDetailDto.getCustomerType();
        if (StringUtils.isNotEmpty(customerType) && "GR".equals(customerType)) {
            List<IpOrderInvoceVo> invoices = ipOrderInvoiceMapper.getOrderInvoice(userId);
            if (invoices != null && invoices.size() > 0) {
                ipOrderCarDetailVo.setInvoiceinfo(invoices.get(0));
            }
        } else {
            List<IpOrderInvoceVo> orderInvoceVos = ipOrderInvoiceMapper.getOrderInvoceDetail(ipOrderDetailDto);
            if (orderInvoceVos != null && orderInvoceVos.size() > 0) {
                ipOrderCarDetailVo.setInvoiceinfo(orderInvoceVos.get(0));
            } else {
                ipOrderCarDetailVo.setInvoiceinfo(new IpOrderInvoceVo());
            }
        }


        return ipOrderCarDetailVo;
    }

    /**
     * 保存与修改订单
     *
     * @param ipOrderAddDto
     * @return
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW,rollbackFor = Exception.class,isolation = Isolation.DEFAULT)
    public IpOrderResultVo saveOrUpdateOrder(IpOrderAddDto ipOrderAddDto) throws Exception {


        IpOrderResultVo orderResultVo = null;
        String customerType = ipOrderAddDto.getCustomerType();
        String ordertype = ipOrderAddDto.getOrder().getOrdertype();
        String userid = ipOrderAddDto.getUserid();
        /**
         * 删除购物车
         */
        if (StringUtils.isNotEmpty(ipOrderAddDto.getCarIds())) {
            if ("GR".equals(customerType)) {
                int carCount = ipcCartService.selectCarIds(ipOrderAddDto.getCarIds());
                if (carCount == 0) {
                    throw new BusinessException(500, "此订单已提交，请勿重复提交");
                }
            }
            ipcCartService.deleteCart(ipOrderAddDto.getCarIds());
        }

        String isSupplier = ipOrderAddDto.getIsSupplier();

        //判断提交类型
        int status = ipOrderAddDto.getStatus();
        Integer drafid = 0;

        //如果为1则为提交
        if (status == 1) {
            drafid = ipOrderAddDto.getDraftId();
            //删除草稿数据
            if (drafid != 0) {
                ipOrderDrafterService.deleOrderDrafter(drafid);
            }
            if (StringUtils.isEmpty(ipOrderAddDto.getInstid())) {
                //验证业务规则，判断是否存在实时价的商品，若为个人采购，则不需验证是否含有实时价的商品
                boolean flag = false;
                //判断是项目还是行政
                if ((StringUtils.isEmpty(customerType) || "XM".equals(customerType)) && "XM".equals(ordertype)) {
                    flag = this.checkRule(ipOrderAddDto);
                }else if ("XZ".equals(ordertype)){
                    this.checkXzOrder(userid);
                    flag=true;
                }else if ("GR".equals(ordertype)) {
                    //如果为个人需验证库存是否充足
                    this.checkGoodsStock(ipOrderAddDto);
                }

                //判断若为实时价、和供应商下单则发起流程，订单状态为审批中
                if (flag || (StringUtils.isNotEmpty(isSupplier) && isSupplier.equals("Y"))) {
                    ipOrderAddDto.setStatusStr("审批中");
                } else {
                    ipOrderAddDto.setAppproedate(DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN));
                    if ("GR".equals(customerType)) {
                        IpOrderDto order = ipOrderAddDto.getOrder();
                        if (order != null) {
                            String totalhsprice = order.getTotalhsprice();
                            ipOrderAddDto.setStatusStr("待付款");
                            if ("0".equals(totalhsprice) || "0.00".equals(totalhsprice)) {
                                ipOrderAddDto.setStatusStr("待发货");
                            }
                            ipOrderAddDto.setExpirationTime(DateUtils.addDateMinute(new Date(), orderConfig.getExpirationtime()));
                        }
                    } else {
                        ipOrderAddDto.setStatusStr("待发货");
                    }

                }

                orderResultVo = ipcOrderService.saveOrUpdateOrder(ipOrderAddDto);
                if (flag || (StringUtils.isNotEmpty(isSupplier) && isSupplier.equals("Y"))) {
                    if (StringUtils.isEmpty(ipOrderAddDto.getInstid())) {
                        String instid = this.startProcess(ipOrderAddDto, orderResultVo, flag,ordertype);
                        IpOrderInstidDto ipOrderInstidDto = new IpOrderInstidDto();
                        ipOrderInstidDto.setInstid(instid);
                        ipOrderInstidDto.setOrderId(orderResultVo.getOrderId());
                        //更新流程实例号
                        ipcOrderService.updateOrderBpmId(ipOrderInstidDto);

                    }
                }
            } else {
                ipOrderAddDto.setStatusStr("审批中");
                orderResultVo = ipcOrderService.saveOrUpdateOrder(ipOrderAddDto);
            }

        } else {
            orderResultVo = new IpOrderResultVo();
            drafid = ipOrderDrafterService.saveOrUpdateDraft(ipOrderAddDto);
            orderResultVo.setDrafid(drafid);
        }
        return orderResultVo;
    }

    /**
     * 校验规则
     * 1.验证项目商品是否可用
     * 2.判断是否存在实时价
     */
    private boolean checkRule(IpOrderAddDto ipOrderAddDto) {

        List<IpOrderGoodDraftsDto> ipOrderGoodDraftsDtos = ipOrderAddDto.getGoods();
        if (ipOrderGoodDraftsDtos == null || ipOrderGoodDraftsDtos.size() == 0) {
            throw new BusinessException(500, "请选择商品之后在提交");
        }
        boolean flag = false;
        //判断商品的采购数量是否在合同内
        IpContractGoodDto ipContractGoodDto = null;
        for (IpOrderGoodDraftsDto ipOrderGoodDraftsDto : ipOrderGoodDraftsDtos) {
            ipContractGoodDto = new IpContractGoodDto();
            String goodsName = ipOrderGoodDraftsDto.getGoodsname();
            String goodsnum = ipOrderGoodDraftsDto.getGoodsnum();
            //数量
            BigDecimal count = new BigDecimal(ipOrderGoodDraftsDto.getCount());
            //价格类型 0 实时价 1 定价
            String pricetype = ipOrderGoodDraftsDto.getPricetype();
            ipContractGoodDto.setContractCode(ipOrderGoodDraftsDto.getContractCode());
            ipContractGoodDto.setGoodsnum(ipOrderGoodDraftsDto.getGoodsnum());
            List<IpContractGoodsVo> contractGoodsVo = ipGoodsContractService.getContractGoodsVo(ipContractGoodDto);
            if (contractGoodsVo == null || contractGoodsVo.size() == 0) {
                throw new BusinessException(500, "您采购的"  + goodsName + "（"+ goodsnum+"）商品在合同清单中不存在");
            }
            IpContractGoodsVo ipContractGoodsVo = contractGoodsVo.get(0);
            BigDecimal noexecutenum = ipContractGoodsVo.getNoexecutenum();
            if (noexecutenum==null){
                noexecutenum=BigDecimal.valueOf(0);
            }
            //判断执行数量是否超未执行
            if (count.compareTo(noexecutenum) == 1) {
                throw new BusinessException(500, "您采购的" + goodsName + "（"+ goodsnum+"）商品在合同清单中不存在或已采购完");
            }
            String goodstotalhsprice = ipOrderGoodDraftsDto.getGoodstotalhsprice();
            String isticket = ipOrderGoodDraftsDto.getIsticket();
            //判断是否是商票
            if ("1".equals(isticket)){
                goodstotalhsprice=ipOrderGoodDraftsDto.getTicketpricetotal();
            }
            if (StringUtils.isEmpty(goodstotalhsprice)) {
                goodstotalhsprice = "0.00";
            }
            BigDecimal goodstotalhspriceBigDecimal = new BigDecimal(goodstotalhsprice);
            if (goodstotalhspriceBigDecimal==null){
                goodstotalhspriceBigDecimal=BigDecimal.valueOf(0);
            }

            //总商品价格
            BigDecimal totalMoney = ipContractGoodsVo.getTotalmoney();
            if (totalMoney != null) {
                //已执行商品价格
                BigDecimal executemoney = ipContractGoodsVo.getExecutemoney();
                if (executemoney==null){
                    executemoney=BigDecimal.valueOf(0);
                }

                BigDecimal add = executemoney.add(goodstotalhspriceBigDecimal);
                //判断商品价格
                if (add.compareTo(totalMoney) == 1) {
                    throw new BusinessException(500, "您采购的" + goodsName + "（"+ goodsnum+"）商品在合同清单中不存在或已采购完");
                }
            }

            //扣减合同商品数量
            ipContractGoodDto.setNum(count);
            ipContractGoodDto.setGoodsMoney(goodstotalhspriceBigDecimal);
            ipGoodsContractService.updateContractGoods(ipContractGoodDto);
            if (pricetype.equals("0")) {
                flag = true;
            }
        }

        return flag;
    }

    /**
     * 验证行政订单，是否配置审批人员
     * @param userid
     */
    private void checkXzOrder(String userid){
        int order = ipAdministrativePersonnelMapper.checkOrder(userid);
        if (order==0){
            throw  new BusinessException(500,"您的订单未匹配到审核人，请联系系统管理员");
        }
    }


    /**
     * 获取行政审批人员
     * @param userid
     * @return
     */
    private String getXzOrderPersion(String userid){
        List<IpAdministrativePersonnelVo> persionList = ipAdministrativePersonnelMapper.getPersionList(userid);
        Set<String> strings=new HashSet<>();
        if (persionList !=null && persionList.size()>0){
            for (IpAdministrativePersonnelVo ipAdministrativePersonnelVo : persionList) {
                strings.add(ipAdministrativePersonnelVo.getApprovalCode());
            }
        }

        return  String.join(",",strings);

    }


    /**
     * 流程发起
     *
     * @param ipOrderAddDto
     * @return
     * @throws Exception
     */
    public String startProcess(IpOrderAddDto ipOrderAddDto, IpOrderResultVo orderResultVo, boolean falg,String ordertype) throws Exception {
        CreateProcessDto createProcessDto = new CreateProcessDto();
        String isSupplierds = "N";
        if (StringUtils.isNotEmpty(ipOrderAddDto.getIsSupplier())) {
            isSupplierds = ipOrderAddDto.getIsSupplier();
        }
        String userid = ipOrderAddDto.getUserid();
        String jglx = "N";
        if (falg) {
            jglx = "Y";
        }
        IpOrderDto ipOrderDto = ipOrderAddDto.getOrder();
        String projectLeadCode=null;
        //若为行政采购从行政人员配置表中获取配置人员
        if ("XZ".equals(ordertype)){
            projectLeadCode=this.getXzOrderPersion(userid);
        }else {
            projectLeadCode=ipOrderDto.getProjectLeaderCode();
        }

        createProcessDto.setProcessName("CGXDSQLC");
        createProcessDto.setUserId(userid);
        createProcessDto.setReqNo(orderResultVo.getOrderNum());
        createProcessDto.setSubject(ipOrderDto.getOrdername());
        Map<String, Object> map = new HashMap<>();
        map.put("CBFZR", ipOrderDto.getCostManageCode());
        map.put("XMFZR", projectLeadCode);
        map.put("JGLX", jglx);
        map.put("ORDERTYPE",ordertype);
        map.put("GYS", isSupplierds);
        createProcessDto.setBusinessData(map);
        logger.info(JSONObject.fromObject(createProcessDto).toString());
        String instid = ipcProcessService.createProcess(createProcessDto);

        return instid;
    }

    /**
     * 零售商品下单验证库存
     *
     * @Param: * @param ipOrderAddDto
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/28 10:42
     */
    private void checkGoodsStock(IpOrderAddDto ipOrderAddDto) throws Exception {
        List<IpOrderGoodDraftsDto> ipOrderGoodDraftsDtos = ipOrderAddDto.getGoods();
        if (ipOrderGoodDraftsDtos == null || ipOrderGoodDraftsDtos.size() == 0) {
            throw new BusinessException(500, "请选择商品之后在提交");
        }
        for (IpOrderGoodDraftsDto ipOrderGoodDraftsDto : ipOrderGoodDraftsDtos) {
            String goodsid = ipOrderGoodDraftsDto.getGoodsid().toString();
            BigDecimal occount = new BigDecimal("0");
            //校验商品库存
            ipcCartService.checkGoodsStock(goodsid, ipOrderGoodDraftsDto.getCount(), occount, ipOrderGoodDraftsDto.getAttrid());
        }
    }

    /**
     * 提交验证库
     *
     * @Param: * @param ipOrderDetailDto
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/11/5 13:47
     */
    @Override
    public void checkStockSubmit(IpOrderDetailDto ipOrderDetailDto) throws Exception {
        //获取商品信息
        List<IpOrderGoodsDetailVo> goods = ipOrderMapper.getOrderGoodsDetail(ipOrderDetailDto);
        if (goods == null || goods.size() == 0) {
            throw new BusinessException(500, "请选择商品之后在提交");
        }
        //判断商品的库存
        for (IpOrderGoodsDetailVo good : goods) {
            String goodsid = good.getGoodsid().toString();
            BigDecimal occount = new BigDecimal("0");
            ipcCartService.checkGoodsStock(goodsid, good.getCount().toString(), occount, good.getAttrid());
        }
    }
}
