package com.ekingwin.bas.cloud.service.impl;

import com.ekingwin.bas.cloud.dao.IpOrderMapper;
import com.ekingwin.bas.cloud.dao.IpcCartMapper;
import com.ekingwin.bas.cloud.dto.IpCarSearchDto;
import com.ekingwin.bas.cloud.dto.IpCarUpdateDto;
import com.ekingwin.bas.cloud.dto.IpCartDeleteDto;
import com.ekingwin.bas.cloud.dto.IpcCartDto;
import com.ekingwin.bas.cloud.entity.IpcCart;
import com.ekingwin.bas.cloud.handler.BusinessException;
import com.ekingwin.bas.cloud.service.IpcCartService;
import com.ekingwin.bas.cloud.util.MapObjUtil;
import com.ekingwin.bas.cloud.util.Result;
import com.ekingwin.bas.cloud.util.StringUtils;
import com.ekingwin.bas.cloud.vo.IpCarDetailVo;
import com.ekingwin.bas.cloud.vo.IpOrderGoodsDetailVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;

/**
 * @program: rcworkspace
 * @description:
 * @author: zhangjie
 * @create: 2019-07-02 18:09
 **/
@Service
@Primary
@Transactional(rollbackFor = Exception.class)
public class IpcCartServiceImpl implements IpcCartService {


    @Autowired
    private IpcCartMapper ipcCartMapper;

    @Autowired
    private IpOrderMapper ipOrderMapper;


    @Value("${goods.baseurl}")
    private String basPath;


    /**
     * 批量删除
     *
     * @param carIds
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class, isolation = Isolation.DEFAULT)
    public void deleteCart(String carIds) {

        ipcCartMapper.deleteByCarIds(carIds);
    }

    /**
     * 获取购物车订单数量
     *
     * @Param: * @param carIds
     * @return: int
     * @Author: zhangjie
     * @Date: 2019/11/6 20:11
     */
    @Override
    public int selectCarIds(String carIds) {

        return ipcCartMapper.selectCarIds(carIds);
    }

    /**
     * 获取购物车数量
     *
     * @param ipCarSearchDto
     * @return
     */
    @Override
    public int getCarCount(IpCarSearchDto ipCarSearchDto) {
        int count = 0;
        if (StringUtils.isNotEmpty(ipCarSearchDto.getCustType()) && ipCarSearchDto.getCustType().equals("Y")) {
            count = ipcCartMapper.getCarOutCount(ipCarSearchDto);
        } else {
            count = ipcCartMapper.getCarCount(ipCarSearchDto);
        }
        return count;
    }

    /**
     * 加入购物车
     * 1.判断商品是否失效
     *
     * @param ipcCartDto
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void saveCar(IpcCartDto ipcCartDto) throws Exception {
        //判断商品的状态
        Integer status = ipcCartMapper.getGoodsStatus(ipcCartDto);
        if (status == null || status != 1) {
            throw new BusinessException(500, "该商品已下架，不能加入购物车！");
        }
        String customerType = ipcCartDto.getCustomerType();
        String goodsId = ipcCartDto.getGoodsid();

        List<IpcCart> ipcCarts = null;
        //判断用户类型是否为外部用户
        if (StringUtils.isNotEmpty(ipcCartDto.getCustType()) && ipcCartDto.getCustType().equals("Y")) {
            ipcCarts = ipcCartMapper.selectOutCarGoods(ipcCartDto);
        } else {
            //判断是否为供应商
            if (StringUtils.isEmpty(ipcCartDto.getIsSupplier())
                    || ipcCartDto.getIsSupplier().equals("N")) {
                ipcCarts = ipcCartMapper.selectCarGoods(ipcCartDto);
            } else {
                ipcCartDto.setLicensor(ipcCartDto.getSupplierid());
                ipcCarts = ipcCartMapper.selectCarGoodsBySuppliers(ipcCartDto);
            }
        }
        //客户类型为内部员工
        if (ipcCartDto != null && "GR".equals(customerType)) {
            BigDecimal ocount = new BigDecimal("0.00");
            if (ipcCarts != null && ipcCarts.size() > 0) {
                IpcCart ipcCartO = ipcCarts.get(0);
                //购物车已有数量
                ocount = new BigDecimal(ipcCartO.getCount());
            }
            this.checkGoodsStock(goodsId, ipcCartDto.getCount(), ocount, ipcCartDto.getAttrid());
        }
        //判断购物车商品是否已存在
        if (ipcCarts != null && ipcCarts.size() > 0) {
            IpcCart ipcCartO = ipcCarts.get(0);
            IpcCart ipcCart = new IpcCart();
            MapObjUtil.copy(ipcCartDto, ipcCart);
            ipcCart.setId(ipcCartO.getId());
            //购物车已有数量
            BigDecimal ocount = new BigDecimal(ipcCartO.getCount());
            //购物车现有数量
            BigDecimal bcount = new BigDecimal(ipcCartDto.getCount());
            //计算总的数量
            BigDecimal total = ocount.add(bcount);
            ipcCart.setCount(total.toString());
            ipcCart.setUserid(ipcCartDto.getUserid());
            ipcCartMapper.updateByPrimaryKeySelective(ipcCart);
        } else {
            IpcCart ipcCart = new IpcCart();
            MapObjUtil.copy(ipcCartDto, ipcCart);
            ipcCart.setUserid(ipcCartDto.getUserid());
            ipcCartMapper.insertSelective(ipcCart);
        }

    }

    /**
     * @param ipCarSearchDto
     * @return
     */
    @Override
    public List<IpCarDetailVo> getCarGoodsList(IpCarSearchDto ipCarSearchDto) {
        List<IpCarDetailVo> carDetailVos = null;
        //判断客户类型，是否是供应商
        if (StringUtils.isNotEmpty(ipCarSearchDto.getCustType()) && ipCarSearchDto.getCustType().equals("Y")) {
            carDetailVos = ipcCartMapper.getOutSupplierList(ipCarSearchDto);
        } else {
            carDetailVos = ipcCartMapper.getSupplierList(ipCarSearchDto);
        }

        for (IpCarDetailVo carDetailVo : carDetailVos) {
            ipCarSearchDto.setSupplierid(carDetailVo.getSupplierid());
            List<IpOrderGoodsDetailVo> ipOrderGoodsDetailVos = ipOrderMapper.getCarGoodsDetail(ipCarSearchDto);
            if (ipOrderGoodsDetailVos != null && ipOrderGoodsDetailVos.size() > 0) {
                for (IpOrderGoodsDetailVo ipOrderGoodsDetailVo : ipOrderGoodsDetailVos) {
                    //判断是否是项目还是零售，若是项目则不返回销售区域
                    if (StringUtils.isNotEmpty(ipCarSearchDto.getCustomerType()) && "GR".equals(ipCarSearchDto.getCustomerType())) {
                        ipOrderGoodsDetailVo.setSalesarea(null);
                    }
                    if (StringUtils.isNotEmpty(ipOrderGoodsDetailVo.getUrlPath())) {
                        ipOrderGoodsDetailVo.setUrlPath(basPath + ipOrderGoodsDetailVo.getUrlPath());
                    }

                }
            }
            carDetailVo.setGoods(ipOrderGoodsDetailVos);
            //失效商品
            List<IpOrderGoodsDetailVo> ipOrderGoodsEffDetailVos = ipOrderMapper.getCarGoodsDetailNoEffect(ipCarSearchDto);
            if (ipOrderGoodsEffDetailVos != null && ipOrderGoodsEffDetailVos.size() > 0) {
                for (IpOrderGoodsDetailVo ipOrderGoodsDetailVo : ipOrderGoodsEffDetailVos) {
                    if (StringUtils.isNotEmpty(ipOrderGoodsDetailVo.getUrlPath())) {
                        ipOrderGoodsDetailVo.setUrlPath(basPath + ipOrderGoodsDetailVo.getUrlPath());
                    }
                }
            }
            carDetailVo.setInvalidgoods(ipOrderGoodsEffDetailVos);

        }

        return carDetailVos;
    }


    /**
     * 清空购物车
     *
     * @param ipCartDeleteDto
     */
    @Override
    public void deleCar(IpCartDeleteDto ipCartDeleteDto) {
        //判断是否清空购物车
        if (ipCartDeleteDto.getType().equals("Y")) {
            ipcCartMapper.deleteCar(ipCartDeleteDto);
        } else {
            List<String> list = ipCartDeleteDto.getId();
            if (list != null && list.size() > 0) {
                String id = String.join(",", list);
                this.deleteCart(id);
            }
        }
    }

    /**
     * @param ipCarUpdateDto
     */
    @Override
    public void updateCar(IpCarUpdateDto ipCarUpdateDto) throws Exception {
        //若为个人验证库存
        if (StringUtils.isNotEmpty(ipCarUpdateDto.getCustomerType()) &&
                "GR".equals(ipCarUpdateDto.getCustomerType())) {
            this.checkCarGoodsStock(ipCarUpdateDto);
        }
        ipcCartMapper.updateCar(ipCarUpdateDto);
    }

    /**
     * 购物车数量验证
     *
     * @Param: * @param ipCarUpdateDto
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/23 11:52
     */
    private void checkCarGoodsStock(IpCarUpdateDto ipCarUpdateDto) throws Exception {
        IpcCart ipcCart = ipcCartMapper.selectByPrimaryKey(ipCarUpdateDto.getId());
        String goodsId = ipcCart.getGoodsid();
        BigDecimal bigDecimal = new BigDecimal("0.00");
        this.checkGoodsStock(goodsId, ipCarUpdateDto.getCount(), bigDecimal, ipcCart.getAttrid());
    }

    /**
     * 校验库存
     *
     * @param count
     * @Param: * @param goodsId
     * @return: void
     * @Author: zhangjie
     * @Date: 2019/10/28 10:40
     */
    @Override
    public Result checkGoodsStock(String goodsId, String count, BigDecimal ocount, String attrid) throws Exception {
        //本次数量
        BigDecimal bDecimal = new BigDecimal(count);
        bDecimal = bDecimal.add(ocount);
        IpOrderGoodsDetailVo goodsStock = null;
        if (StringUtils.isNotEmpty(attrid)) {
            goodsStock = ipcCartMapper.getGoodsStockAttrid(goodsId, attrid);
        } else {
            //获取库存
            goodsStock = ipcCartMapper.getGoodsStock(goodsId);
        }


        if (!"1".equals(goodsStock.getState())) {
            return Result.error(goodsStock.getGoodsname() + "商品已下架");
        }
        BigDecimal stock = new BigDecimal(goodsStock.getStock());
        if (stock.compareTo(bDecimal) == -1) {
            return Result.error(goodsStock.getGoodsname() + "商品库存不足");
        }
        return null;
    }


}
